<?php
// Register main menu for Tipax
add_action('admin_menu', 'register_tipax_shipping_menu');

function register_tipax_shipping_menu() {
    add_menu_page(
        'Tipax Shipping',             // Page title
        'Tipax Shipping',             // Menu title
        'manage_options',             // Capability
        'tipax-api-config',    // Menu slug
        'tipax_api_config_page',  // Callback function
        'dashicons-admin-generic',    // Icon
        56                            // Position
    );
    
    add_submenu_page(
        null,
        'Test Connection',
        'Test Connection',
        'manage_options',
        'tipax-test-connection',
        'tipax_test_connection_page'
    );
    
    add_submenu_page(
        null,
        'Addresses',
        'Addresses',
        'manage_options',
        'tipax-addresses',
        'tipax_addresses_page'
    );
    
    add_submenu_page(
        null,
        'Orders',
        'Orders',
        'manage_options',
        'tipax-orders',
        'tipax_orders_page'
    );
    
    add_submenu_page(
        null,
        'Wallet',
        'Wallet',
        'manage_options',
        'tipax-wallet',
        'tipax_wallet'
    );
    
    add_submenu_page(
        null,
        'Settings',
        'Settings',
        'manage_options',
        'tipax-settings',
        'tipax_settings_page'
    );
    
    add_submenu_page(
        null,
        'Logs',
        'Logs',
        'manage_options',
        'tipax-logs',
        'tipax_logs_page'
    );
}

function tipax_api_config_page() {
    tipax_admin_tabs('tipax-api-config');
    if (!class_exists('WC_Tipax_Shipping_Method')) {
        require_once plugin_dir_path(__FILE__) . 'classes/WC_Tipax_Shipping_Method.php'; // adjust the path if needed
    }
    // Initialize the WC_Tipax_Shipping_Method class to access settings
    $shipping_method = new WC_Tipax_Shipping_Method();
    $shipping_method->init(); // This loads the settings

    // Save settings if the form is submitted (this triggers WooCommerce's settings API)
    if ( isset( $_POST['woocommerce_tipax_shipping_method_username'], $_POST['woocommerce_tipax_shipping_method_password'], $_POST['woocommerce_tipax_shipping_method_apiKey'] ) ) {
        // Save settings to WooCommerce's options table
        $shipping_method->settings['username'] = sanitize_text_field( $_POST['woocommerce_tipax_shipping_method_username'] );
        $shipping_method->settings['password'] = sanitize_text_field( $_POST['woocommerce_tipax_shipping_method_password'] );
        $shipping_method->settings['apiKey']  = sanitize_text_field( $_POST['woocommerce_tipax_shipping_method_apiKey'] );
        
        // Ensure 'enabled' is saved as 'yes' if checkbox is checked, otherwise 'no'
        $shipping_method->settings['enabled']  = isset( $_POST['woocommerce_tipax_shipping_method_enabled'] ) ? 'yes' : 'no';  

        // Save settings using WooCommerce's process_admin_options method
        $shipping_method->process_admin_options();

        // After saving, reload the settings so the form reflects the new values
        $shipping_method->init(); // This will reload the settings

        echo '<div class="updated"><p>' . __( 'Settings saved.', 'delivery-service-tipax-shipping' ) . '</p></div>';
    }
    // Retrieve saved settings from the class
    $tipax_username = isset($shipping_method->settings['username']) ? $shipping_method->settings['username'] : '';
    $tipax_password = isset($shipping_method->settings['password']) ? $shipping_method->settings['password'] : '';
    $tipax_api_key  = isset($shipping_method->settings['apiKey']) ? $shipping_method->settings['apiKey'] : '';
    $tipax_enabled  = $shipping_method->get_option('enabled');  // Use get_option to get 'enabled' properly


    ?>
    <h2>API Configuration</h2>
    <form method="post" action="">
        <table class="form-table">
            <tr valign="top">
                <th scope="row">
                    <label for="enabled"><?php _e( 'Enable/Disable', 'delivery-service-tipax-shipping' ); ?></label>
                </th>
                <td>
                    <input type="checkbox" name="woocommerce_tipax_shipping_method_enabled" id="enabled" value="1" <?php checked( $tipax_enabled, 'yes' ); ?> />
                    <p class="description"><?php _e( 'Enable this shipping method', 'delivery-service-tipax-shipping' ); ?></p>
                </td>
            </tr>

            <tr valign="top">
                <th scope="row">
                    <label for="username"><?php _e( 'Username', 'delivery-service-tipax-shipping' ); ?></label>
                </th>
                <td>
                    <input type="text" name="woocommerce_tipax_shipping_method_username" id="tipaxusername" value="<?php echo esc_attr( $tipax_username ); ?>" class="regular-text" required />
                    <p class="description"><?php _e( 'Username of your Tipax account', 'delivery-service-tipax-shipping' ); ?></p>
                </td>
            </tr>

            <tr valign="top">
                <th scope="row">
                    <label for="password"><?php _e( 'Password', 'delivery-service-tipax-shipping' ); ?></label>
                </th>
                <td>
                    <input type="password" name="woocommerce_tipax_shipping_method_password" id="password" value="<?php echo esc_attr( $tipax_password ); ?>" class="regular-text" required />
                    <p class="description"><?php _e( 'Password of your Tipax account', 'delivery-service-tipax-shipping' ); ?></p>
                </td>
            </tr>

            <tr valign="top">
                <th scope="row">
                    <label for="apiKey"><?php _e( 'API Key', 'delivery-service-tipax-shipping' ); ?></label>
                </th>
                <td>
                    <input type="text" name="woocommerce_tipax_shipping_method_apiKey" id="apiKey" value="<?php echo esc_attr( $tipax_api_key ); ?>" class="regular-text" required />
                    <p class="description"><?php _e( 'This Tipax API Key is located in your profile page', 'delivery-service-tipax-shipping' ); ?></p>
                </td>
            </tr>
        </table>

        <?php submit_button( 'Save Settings' ); ?>
    </form>
    <?php
}

// Hook to handle the AJAX request for testing the connection
add_action('wp_ajax_test_connection', 'handle_test_connection_ajax');

function handle_test_connection_ajax() {
    // Instantiate the WC_Tipax_V2 class to get the test connection method
    $shipping_method_v2 = new WC_Tipax_V2();
    
    // Call the checkApiKey method to test the connection
    $response_message = $shipping_method_v2->checkApiKey();

    // Return the response message to the AJAX request
    echo $response_message;
    
    // Always call wp_die() to properly terminate AJAX requests in WordPress
    wp_die();
}

function tipax_test_connection_page() {
    tipax_admin_tabs('tipax-test-connection'); // Optional tab

    echo '<h2>Test API & Environment</h2>';
    echo '<p>Click the button below to run a full diagnostics test.</p>';
    echo '<button id="tipax_run_diagnostics" class="button button-primary">Run Diagnostics</button>';
    echo '<div id="tipax_diagnostics_results" style="margin-top:20px;"></div>';

    ?>
    <script>
        jQuery(document).ready(function($) {
            $('#tipax_run_diagnostics').on('click', function() {
                $('#tipax_diagnostics_results').html('<p>Testing in progress...</p>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'tipax_run_diagnostics'
                    },
                    success: function(response) {
                        $('#tipax_diagnostics_results').html(response);
                    },
                    error: function() {
                        $('#tipax_diagnostics_results').html('<div class="notice notice-error"><p>Something went wrong while testing.</p></div>');
                    }
                });
            });
        });
    </script>
    <?php
}

add_action('wp_ajax_tipax_run_diagnostics', 'tipax_run_diagnostics');

function tipax_run_diagnostics() {
    $results = [];
    
    if (!class_exists('WC_Tipax_Shipping_Method')) {
        require_once plugin_dir_path(__FILE__) . 'classes/WC_Tipax_Shipping_Method.php'; // adjust the path if needed
    }
    
    // 1. Tipax Shipping Method Enabled Check
    try {
        $shipping_method = new WC_Tipax_Shipping_Method(); // Make sure this class is available and autoloaded
        $shipping_method->init(); // Loads settings
    
        $enabled_status = isset($shipping_method->settings['enabled']) ? $shipping_method->settings['enabled'] : 'no';
    
        $results[] = [
            'name' => 'Tipax Shipping Method Status',
            'status' => ($enabled_status === 'yes') ? '✅ Enabled' : '❌ Disabled',
            'message' => 'Shipping method is currently ' . (($enabled_status === 'yes') ? 'active' : 'inactive')
        ];
    } catch (Throwable $e) {
        $results[] = [
            'name' => 'Tipax Shipping Method Status',
            'status' => '❌ Error',
            'message' => 'Could not read settings: ' . esc_html($e->getMessage())
        ];
    }


    // 2. Ping
    $ping = @fsockopen("omapi.tipax.ir", 443, $errno, $errstr, 5);
    $results[] = [
        'name' => 'Ping Tipax Server (443)',
        'status' => $ping ? '✅ Success' : '❌ Failed',
        'message' => $ping ? 'Connected successfully' : "Error: $errstr ($errno)"
    ];
    if ($ping) fclose($ping);

    // 3. SSL Handshake
    $ch = curl_init("https://omapi.tipax.ir");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_exec($ch);
    $ssl_error = curl_error($ch);
    $results[] = [
        'name' => 'TLS/SSL Handshake',
        'status' => empty($ssl_error) ? '✅ Success' : '❌ Failed',
        'message' => empty($ssl_error) ? 'Handshake completed' : $ssl_error
    ];
    curl_close($ch);

    // 4. WordPress
    $wp_version = get_bloginfo('version');
    $results[] = [
        'name' => 'WordPress Version',
        'status' => version_compare($wp_version, '5.8', '>=') ? '✅ OK' : '❌ Too old',
        'message' => 'Current version: ' . $wp_version
    ];

    // 5. WooCommerce
    if (class_exists('WooCommerce')) {
        $wc_version = WC()->version;
        $results[] = [
            'name' => 'WooCommerce Version',
            'status' => version_compare($wc_version, '6.0', '>=') ? '✅ OK' : '❌ Too old',
            'message' => 'Current version: ' . $wc_version
        ];
    } else {
        $results[] = [
            'name' => 'WooCommerce Check',
            'status' => '❌ Not Installed',
            'message' => 'WooCommerce plugin is not active.'
        ];
    }
    
    // 6. PHP
    if (version_compare(PHP_VERSION, '7.4', '>=')) {
        $results[] = [
            'name' => 'PHP Version',
            'status' => '✅ OK',
            'message' => 'PHP version is ' . PHP_VERSION
        ];
    } else {
        $results[] = [
            'name' => 'PHP Version',
            'status' => '❌ Too Old',
            'message' => 'Current PHP version is ' . PHP_VERSION . '. Please upgrade to 7.4 or higher.'
        ];
    }

    // 7. curl
    if (!function_exists('curl_version')) {
        $results[] = [
            'name' => 'PHP cURL Extension',
            'status' => '❌ Not Installed',
            'message' => 'Your server is missing the cURL extension. API requests may fail.'
        ];
    } else {
        $results[] = [
            'name' => 'PHP cURL Extension',
            'status' => '✅ Installed',
            'message' => 'cURL is available and used by WordPress for HTTP requests.'
        ];
    }
    
    // 8. Tipax Addresses Fetch Test
    try {
        if (!class_exists('WC_Tipax_V2')) {
            require_once plugin_dir_path(__FILE__) . 'includes/classes/WC_Tipax_V2.php'; // Adjust as needed
        }
    
        $shipping_method = new WC_Tipax_V2();
    
        // Clear cache and force fresh fetch
        delete_transient('tipax_addresses');
        $addresses_result = $shipping_method->loadAddresses();
    
        if ($addresses_result && isset($addresses_result->address)) {
            $addresses = is_array($addresses_result->address) ? $addresses_result->address : [$addresses_result->address];
            $first = $addresses[0] ?? null;
    
            // ✅ Save success flag
            update_option('tipax_has_address', 'yes');
    
            $results[] = [
                'name' => 'Tipax Address Fetch',
                'status' => '✅ Success',
                'message' => 'Loaded ' . count($addresses) . ' address(es). Example: ' . esc_html($first->fullAddress ?? 'N/A')
            ];
        } else {
            // ❌ Save failure flag
            update_option('tipax_has_address', 'no');
    
            $results[] = [
                'name' => 'Tipax Address Fetch',
                'status' => '❌ Failed',
                'message' => 'No addresses returned from API or response malformed.'
            ];
        }
    } catch (Throwable $e) {
        update_option('tipax_has_address', 'no');
    
        $results[] = [
            'name' => 'Tipax Address Fetch',
            'status' => '❌ Error',
            'message' => $e->getMessage()
        ];
    }

    // 9. API Check
    try {
        $shipping_method = new WC_Tipax_V2();
        $api_response = $shipping_method->checkApiKey();
        $ok = (stripos($api_response, 'موفق') !== false || stripos($api_response, 'Success') !== false);
        update_option('tipax_test_success', $ok ? 'yes' : 'no');
        $results[] = [
            'name' => 'Tipax API Key Validation',
            'status' => $ok ? '✅ Success' : '❌ Failed',
            'message' => $api_response
        ];
    } catch (Exception $e) {
        $results[] = [
            'name' => 'Tipax API Key Validation',
            'status' => '❌ Error',
            'message' => $e->getMessage()
        ];
        update_option('tipax_test_success', 'no');
    }

    // Output results as table HTML
    echo '<table class="widefat striped">';
    echo '<thead><tr><th>Check</th><th>Status</th><th>Message</th></tr></thead><tbody>';
    foreach ($results as $row) {
        echo '<tr>';
        echo '<td>' . esc_html($row['name']) . '</td>';
        echo '<td>' . esc_html($row['status']) . '</td>';
        echo '<td>' . esc_html($row['message']) . '</td>';
        echo '</tr>';
    }
    echo '</tbody></table>';

    wp_die();
}

add_action('wp_ajax_test_connection_ajax', function() {
    $shipping_method = new WC_Tipax_V2();
    $result = $shipping_method->checkApiKey();

    if (stripos($result, 'موفق') !== false || stripos($result, 'Success') !== false) {
        update_option('tipax_test_success', 'yes');
        wp_send_json(['success' => true, 'message' => '✅ اتصال موفق بود.']);
    } else {
        update_option('tipax_test_success', 'no');
        wp_send_json(['success' => false, 'message' => '❌ اتصال موفق نبود.']);
    }

    wp_die();
});

function tipax_admin_tabs($current = 'connection') {
    $tabs = [
        'tipax-api-config' => 'Connection',
        'tipax-test-connection' => 'Test Connection',
        'tipax-addresses' => 'Addresses',
        'tipax-orders' => 'Orders',
        'tipax-wallet' => 'Wallet',
        'tipax-settings' => 'Settings',
        'tipax-logs' => 'Logs',
    ];

    echo '<h2 class="nav-tab-wrapper">';
    foreach ($tabs as $slug => $name) {
        $class = ($slug == $current) ? ' nav-tab-active' : '';
        echo '<a class="nav-tab' . $class . '" href="?page=' . $slug . '">' . $name . '</a>';
    }
    echo '</h2>';
}


function tipax_is_connection_validated() {
    $username = get_option('woocommerce_tipax_shipping_method_username');
    $password = get_option('woocommerce_tipax_shipping_method_password');
    $apiKey = get_option('woocommerce_tipax_shipping_method_apiKey');
    $test_success = get_option('tipax_test_success', false);
    return $test_success === 'yes';
}

function tipax_wallet() {
    tipax_admin_tabs('tipax-wallet');
    if (!tipax_is_connection_validated()) {
        echo '<div class="notice notice-error"><p>برای دسترسی به این بخش ابتدا باید <a href="' . admin_url('admin.php?page=tipax-api-config') . '">تنظیمات اتصال</a> را کامل کرده و اتصال را <a href="' . admin_url('admin.php?page=tipax-test-connection') . '">آزمایش</a> کنید.</p></div>';
        return;
    }
    if (get_option('tipax_has_address') !== 'yes') {
        echo '<div class="notice notice-warning"><p>برای استفاده از این بخش باید حداقل یک آدرس ثبت شده در حساب تیپاکس شما باشد. لطفاً به <a href="' . admin_url('admin.php?page=tipax-addresses') . '">بخش آدرس‌ها</a> مراجعه کنید.</p></div>';
        return;
    }
    $remaining_assets = get_wallet_remaining_asset();
    echo '<h2>کیف پول تیپاکس</h2>';
    echo '<p>موجودی کیف پول: ' . $remaining_assets . ' ریال</p>';

    echo '<form method="post" action="">';
    echo '<label for="recharge_amount">مقدار شارژ به ریال:</label>';
    echo '<input type="number" id="recharge_amount" name="recharge_amount" required>';
    echo '<input type="submit" name="submit_recharge" value="شارژ کیف پول">';
    echo '</form>';

    if (isset($_POST['submit_recharge'])) {
        recharge_wallet($_POST['recharge_amount']);
    }
}

function get_wallet_remaining_asset() {
    $request = new Request();
    $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);
    $data = [];
    $response = $request->apiRequest('Customers/Wallet', $data, "GET", $accessToken);
    if(is_numeric($response)){
        return number_format($response);
    } else {
        return 'ERROR';
    }
}

function recharge_wallet($amount) {
    if(is_numeric($amount) && $amount > 1000){
        $request = new Request();
        $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);
        $data = new stdClass;
        $data->amount = $amount;
        $data->frontCallBackUrl = admin_url().'admin.php?page=tipax-shipping-method';
        
        $response = $request->apiRequest('Customers/RechargeWallet', $data, "POST", $accessToken);
        header("Location: ".$response->paymentURL);
    } else {
        echo 'مقدار شارژ معتبر نیست.';
    }
}

/**
 * Ensure WP Media Uploader is available on Tipax Settings page
 */
add_action('admin_enqueue_scripts', function ($hook) {
    if (isset($_GET['page']) && $_GET['page'] === 'tipax-settings') {
        wp_enqueue_media(); // ✅ This loads wp.media
    }
});


function tipax_settings_page() {
    // 0) Detect conflicting plugin (persian-woocommerce-shipping)
    include_once ABSPATH . 'wp-admin/includes/plugin.php';
    $is_pws_active = function_exists('is_plugin_active')
        && is_plugin_active('persian-woocommerce-shipping/woocommerce-shipping.php');

    tipax_admin_tabs('tipax-settings');

    if (!tipax_is_connection_validated()) {
        echo '<div class="notice notice-error"><p>برای دسترسی به این بخش ابتدا باید <a href="' . admin_url('admin.php?page=tipax-api-config') . '">تنظیمات اتصال</a> را کامل کرده و اتصال را <a href="' . admin_url('admin.php?page=tipax-test-connection') . '">آزمایش</a> کنید.</p></div>';
        return;
    }

    if (get_option('tipax_has_address') !== 'yes') {
        echo '<div class="notice notice-warning"><p>برای استفاده از این بخش باید حداقل یک آدرس ثبت شده در حساب تیپاکس شما باشد. لطفاً به <a href="' . admin_url('admin.php?page=tipax-addresses') . '">بخش آدرس‌ها</a> مراجعه کنید.</p></div>';
        return;
    }

    // Optional: show why the checkbox is locked
    if ($is_pws_active) {
        echo '<div class="notice notice-info"><p>به‌دلیل فعال بودن افزونه <code>persian-woocommerce-shipping</code>، گزینه «شهرهای تیپاکس» غیر فعال و همواره خاموش است.</p></div>';
    }

    $option_key = 'woocommerce_tipax_shipping_method_settings';

    // 1) Handle Save
    if (isset($_POST['tipax_save_settings'])) {
        $settings = get_option($option_key, []);

        // Force cities = 'no' while conflicting plugin is active
        if ($is_pws_active) {
            $settings['cities'] = 'no';
        } else {
            $settings['cities'] = isset($_POST['woocommerce_tipax_shipping_method_cities']) ? 'yes' : 'no';
        }
        $settings['show_product_table'] = isset($_POST['tipax_show_product_table']) ? 'yes' : 'no';
        $settings['logo_url'] = esc_url_raw($_POST['tipax_logo_url'] ?? '');
        $settings['phone'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_phone'] ?? '');
        $settings['name'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_name'] ?? '');
        $settings['checkout_method'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_checkout_method'] ?? '');
        $settings['collection_time'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_collection_time'] ?? '');
        $settings['inner_flat_price'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_inner_flat_price'] ?? '');
        $settings['outer_flat_price'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_outer_flat_price'] ?? '');
        $settings['inner_min_cart_free_shipping'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_inner_min_cart_free_shipping'] ?? '');
        $settings['outer_min_cart_free_shipping'] = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_outer_min_cart_free_shipping'] ?? '');

        // New: packtype (select) + averages
        $allowed_packtypes = ['10', '20', '50']; // 10: پاکت, 20: بسته, 50: مینی پک
        $posted_packtype = sanitize_text_field($_POST['woocommerce_tipax_shipping_method_packtype'] ?? '');
        $settings['packtype'] = in_array($posted_packtype, $allowed_packtypes, true) ? $posted_packtype : '20';

        $settings['avg_weight'] = isset($_POST['woocommerce_tipax_shipping_method_avg_weight']) && is_numeric($_POST['woocommerce_tipax_shipping_method_avg_weight'])
            ? (float) $_POST['woocommerce_tipax_shipping_method_avg_weight'] : '';
        $settings['avg_length'] = isset($_POST['woocommerce_tipax_shipping_method_avg_length']) && is_numeric($_POST['woocommerce_tipax_shipping_method_avg_length'])
            ? (float) $_POST['woocommerce_tipax_shipping_method_avg_length'] : '';
        $settings['avg_width']  = isset($_POST['woocommerce_tipax_shipping_method_avg_width']) && is_numeric($_POST['woocommerce_tipax_shipping_method_avg_width'])
            ? (float) $_POST['woocommerce_tipax_shipping_method_avg_width'] : '';
        $settings['avg_height'] = isset($_POST['woocommerce_tipax_shipping_method_avg_height']) && is_numeric($_POST['woocommerce_tipax_shipping_method_avg_height'])
            ? (float) $_POST['woocommerce_tipax_shipping_method_avg_height'] : '';

        update_option($option_key, $settings);
        echo '<div class="notice notice-success"><p>تنظیمات ذخیره شد.</p></div>';
    }

    // 2) Get saved values
    $settings = get_option($option_key, []);
    $cities_enabled = !$is_pws_active && (isset($settings['cities']) && $settings['cities'] === 'yes');
    $phone = esc_attr($settings['phone'] ?? '');
    $name = esc_attr($settings['name'] ?? '');
    $checkout_method = $settings['checkout_method'] ?? '20';
    $collection_time = $settings['collection_time'] ?? '1';
    $inner_flat_price = esc_attr($settings['inner_flat_price'] ?? '');
    $outer_flat_price = esc_attr($settings['outer_flat_price'] ?? '');
    $inner_min_cart_free = esc_attr($settings['inner_min_cart_free_shipping'] ?? '');
    $outer_min_cart_free = esc_attr($settings['outer_min_cart_free_shipping'] ?? '');
    $show_product_table = ($settings['show_product_table'] ?? 'yes') === 'yes';

    // New: load packtype + averages
    $packtype   = esc_attr($settings['packtype']   ?? '20'); // default بسته
    $avg_weight = esc_attr($settings['avg_weight'] ?? '');
    $avg_length = esc_attr($settings['avg_length'] ?? '');
    $avg_width  = esc_attr($settings['avg_width']  ?? '');
    $avg_height = esc_attr($settings['avg_height'] ?? '');

    $checkout_methods = [
        "20"  => __('Post Pay (After Payment)', 'delivery-service-tipax-shipping'),
        "30"  => __('Cash on Delivery (COD)', 'delivery-service-tipax-shipping'),
        "50"  => __('Cash with your wallet in tipax', 'delivery-service-tipax-shipping'),
        "10"  => __('Cash with receiver in invoice', 'delivery-service-tipax-shipping'),
        "100" => __('Cash with receiver with flat rate', 'delivery-service-tipax-shipping'),
    ];

    $collection_times = [
        "1" => __('The same day', 'delivery-service-tipax-shipping'),
    ];

    // New: packtype options (Persian)
    $packtype_options = [
        '10' => 'پاکت',
        '20' => 'بسته',
        '50' => 'مینی پک',
    ];

    echo '<h2>تنظیمات</h2>';
    echo '<form method="post" action="">';
    echo '<table class="form-table">';

    // 3) Render the checkbox, disabled if conflicting plugin is active
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_cities">شهرهای تیپاکس</label></th>';
    echo '<td>';
    echo '<label>';
    echo '<input type="checkbox" name="woocommerce_tipax_shipping_method_cities" id="woocommerce_tipax_shipping_method_cities" value="1" '
        . checked($cities_enabled, true, false)
        . ($is_pws_active ? ' disabled="disabled"' : '')
        . ' > فقط شهرهای پشتیبانی‌شده تیپاکس';
    echo '</label>';
    if ($is_pws_active) {
        echo '<p class="description">این گزینه به علت تداخل با افزونه Persian WooCommerce Shipping غیر فعال شده است.</p>';
    }
    echo '</td>';
    echo '</tr>';
// --- New: Logo upload ---
$logo_url = esc_url($settings['logo_url'] ?? '');
echo '<tr>';
echo '<th scope="row"><label for="tipax_logo_url">لوگوی چاپ تیپاکس</label></th>';
echo '<td>';
echo '<input type="hidden" name="tipax_logo_url" id="tipax_logo_url" value="' . $logo_url . '" />';
echo '<img id="tipax_logo_preview" src="' . ($logo_url ?: 'https://via.placeholder.com/150x50?text=No+Logo') . '" style="max-width:150px;display:block;margin-bottom:6px;border:1px solid #ccc;padding:2px;" />';
echo '<button type="button" class="button" id="tipax_logo_upload_btn">انتخاب لوگو</button> ';
echo '<button type="button" class="button" id="tipax_logo_remove_btn">حذف</button>';
echo '<p class="description">این لوگو در چاپ فاکتور و برچسب تیپاکس نمایش داده می‌شود.</p>';
echo '</td>';
echo '</tr>';

    // phone
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_phone">تلفن فرستنده</label></th>';
    echo '<td><input type="text" name="woocommerce_tipax_shipping_method_phone" id="woocommerce_tipax_shipping_method_phone" class="regular-text" value="' . $phone . '" /></td>';
    echo '</tr>';

    // name
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_name">نام کامل فرستنده</label></th>';
    echo '<td><input type="text" name="woocommerce_tipax_shipping_method_name" id="woocommerce_tipax_shipping_method_name" class="regular-text" value="' . $name . '" /></td>';
    echo '</tr>';

    // checkout method
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_checkout_method">روش تسویه حساب</label></th>';
    echo '<td><select name="woocommerce_tipax_shipping_method_checkout_method" id="woocommerce_tipax_shipping_method_checkout_method">';
    foreach ($checkout_methods as $value => $label) {
        echo '<option value="' . esc_attr($value) . '" ' . selected($checkout_method, $value, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select></td>';
    echo '</tr>';

    // Flat rate fields (hidden unless method=100)
    echo '<tr class="tipax-flat-rate-prices" style="display: none;">';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_inner_flat_price">قیمت درون شهری</label></th>';
    echo '<td><input type="number" name="woocommerce_tipax_shipping_method_inner_flat_price" id="woocommerce_tipax_shipping_method_inner_flat_price" class="regular-text" value="' . $inner_flat_price . '" /></td>';
    echo '</tr>';

    echo '<tr class="tipax-flat-rate-prices" style="display: none;">';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_outer_flat_price">قیمت بین شهری</label></th>';
    echo '<td><input type="number" name="woocommerce_tipax_shipping_method_outer_flat_price" id="woocommerce_tipax_shipping_method_outer_flat_price" class="regular-text" value="' . $outer_flat_price . '" /></td>';
    echo '</tr>';

    echo '<tr class="tipax-flat-rate-prices" style="display: none;">';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_inner_min_cart_free_shipping">حداقل مبلغ سبد برای ارسال رایگان درون شهری</label></th>';
    echo '<td><input type="number" name="woocommerce_tipax_shipping_method_inner_min_cart_free_shipping" id="woocommerce_tipax_shipping_method_inner_min_cart_free_shipping" class="regular-text" value="' . $inner_min_cart_free . '" /></td>';
    echo '</tr>';

    echo '<tr class="tipax-flat-rate-prices" style="display: none;">';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_outer_min_cart_free_shipping">حداقل مبلغ سبد برای ارسال رایگان بین شهری</label></th>';
    echo '<td><input type="number" name="woocommerce_tipax_shipping_method_outer_min_cart_free_shipping" id="woocommerce_tipax_shipping_method_outer_min_cart_free_shipping" class="regular-text" value="' . $outer_min_cart_free . '" /></td>';
    echo '</tr>';
// --- New: نوع بسته‌بندی (select) ---
/*
    // --- New: نوع بسته‌بندی (select) ---
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_packtype">نوع بسته‌بندی</label></th>';
    echo '<td><select name="woocommerce_tipax_shipping_method_packtype" id="woocommerce_tipax_shipping_method_packtype">';
    foreach ($packtype_options as $value => $label) {
        echo '<option value="' . esc_attr($value) . '" ' . selected($packtype, $value, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select></td>';
    echo '</tr>';
*/
    // --- New: averages (Persian labels) ---
    // متوسط وزن محصول (کیلوگرم)
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_avg_weight">متوسط وزن محصول (کیلوگرم)</label></th>';
    echo '<td><input type="number" step="0.01" min="0" name="woocommerce_tipax_shipping_method_avg_weight" id="woocommerce_tipax_shipping_method_avg_weight" class="regular-text" value="' . $avg_weight . '" placeholder="مثلاً 0.5" /></td>';
    echo '</tr>';

    // متوسط طول محصول (سانتی‌متر)
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_avg_length">متوسط طول محصول (سانتی‌متر)</label></th>';
    echo '<td><input type="number" step="0.1" min="0" name="woocommerce_tipax_shipping_method_avg_length" id="woocommerce_tipax_shipping_method_avg_length" class="regular-text" value="' . $avg_length . '" placeholder="مثلاً 20" /></td>';
    echo '</tr>';

    // متوسط عرض محصول (سانتی‌متر)
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_avg_width">متوسط عرض محصول (سانتی‌متر)</label></th>';
    echo '<td><input type="number" step="0.1" min="0" name="woocommerce_tipax_shipping_method_avg_width" id="woocommerce_tipax_shipping_method_avg_width" class="regular-text" value="' . $avg_width . '" placeholder="مثلاً 15" /></td>';
    echo '</tr>';

    // متوسط ارتفاع محصول (سانتی‌متر)
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_avg_height">متوسط ارتفاع محصول (سانتی‌متر)</label></th>';
    echo '<td><input type="number" step="0.1" min="0" name="woocommerce_tipax_shipping_method_avg_height" id="woocommerce_tipax_shipping_method_avg_height" class="regular-text" value="' . $avg_height . '" placeholder="مثلاً 10" /></td>';
    echo '</tr>';

    // collection time
    echo '<tr>';
    echo '<th scope="row"><label for="woocommerce_tipax_shipping_method_collection_time">زمان جمع‌آوری</label></th>';
    echo '<td><select name="woocommerce_tipax_shipping_method_collection_time" id="woocommerce_tipax_shipping_method_collection_time">';
    foreach ($collection_times as $value => $label) {
        echo '<option value="' . esc_attr($value) . '" ' . selected($collection_time, $value, false) . '>' . esc_html($label) . '</option>';
    }
    echo '</select></td>';
    echo '</tr>';
// --- New: Show product table ---
echo '<tr>';
echo '<th scope="row"><label for="tipax_show_product_table">نمایش جدول محصولات در چاپ لیبل سفارش</label></th>';
echo '<td>';
echo '<label><input type="checkbox" name="tipax_show_product_table" id="tipax_show_product_table" value="1" ' . checked($show_product_table, true, false) . '> نمایش جزئیات محصولات در فاکتور چاپی</label>';
echo '</td>';
echo '</tr>';

    echo '</table>';
    submit_button('ذخیره تنظیمات', 'primary', 'tipax_save_settings');
    echo '</form>';

    ?>
    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const checkoutSelect = document.getElementById('woocommerce_tipax_shipping_method_checkout_method');
        const flatRateFields = document.querySelectorAll('.tipax-flat-rate-prices');

        function toggleFlatRateFields() {
            if (checkoutSelect.value === '100') {
                flatRateFields.forEach(el => el.style.display = 'table-row');
            } else {
                flatRateFields.forEach(el => el.style.display = 'none');
            }
        }

        checkoutSelect.addEventListener('change', toggleFlatRateFields);
        toggleFlatRateFields();
    });
    // --- Tipax logo upload ---
const logoBtn = document.getElementById('tipax_logo_upload_btn');
const removeBtn = document.getElementById('tipax_logo_remove_btn');
const logoInput = document.getElementById('tipax_logo_url');
const preview = document.getElementById('tipax_logo_preview');
let frame;

if (logoBtn) {
    logoBtn.addEventListener('click', function(e) {
        e.preventDefault();
        if (frame) { frame.open(); return; }
        frame = wp.media({
            title: 'انتخاب لوگو',
            button: { text: 'استفاده از این لوگو' },
            multiple: false
        });
        frame.on('select', function() {
            const attachment = frame.state().get('selection').first().toJSON();
            logoInput.value = attachment.url;
            preview.src = attachment.url;
        });
        frame.open();
    });
}

if (removeBtn) {
    removeBtn.addEventListener('click', function(e) {
        e.preventDefault();
        logoInput.value = '';
        preview.src = 'https://via.placeholder.com/150x50?text=No+Logo';
    });
}

    </script>
    <?php
}





function tipax_addresses_page() {
    tipax_admin_tabs('tipax-addresses');

    if (!tipax_is_connection_validated()) {
        echo '<div class="notice notice-error"><p>برای دسترسی به این بخش ابتدا باید <a href="' . admin_url('admin.php?page=tipax-api-config') . '">تنظیمات اتصال</a> را کامل کرده و اتصال را <a href="' . admin_url('admin.php?page=tipax-test-connection') . '">آزمایش</a> کنید.</p></div>';
        return;
    }

    echo '<h2>آدرس‌های ذخیره شده از تیپاکس</h2>';
    echo '<button id="update_tipax_addresses" class="button button-primary">بروزرسانی آدرس‌ها از تیپاکس</button>';
    echo '<div id="tipax_addresses_result" style="margin-top: 20px;"></div>';

    ?>
    <script>
        jQuery(document).ready(function($) {
            function loadAddresses() {
                $('#tipax_addresses_result').html('<p>در حال بارگیری...</p>');
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'tipax_reload_addresses'
                    },
                    success: function(response) {
                        $('#tipax_addresses_result').html(response);
                    },
                    error: function() {
                        $('#tipax_addresses_result').html('<div class="notice notice-error"><p>خطا در بارگیری آدرس‌ها</p></div>');
                    }
                });
            }

            // Load addresses initially
            loadAddresses();

            // Reload when button is clicked
            $('#update_tipax_addresses').on('click', function() {
                loadAddresses();
            });
        });
    </script>
    <?php
}


add_action('wp_ajax_tipax_reload_addresses', 'tipax_reload_addresses');

function tipax_reload_addresses() {
    delete_transient('tipax_addresses');
    try {
        $shipping_method = new WC_Tipax_V2();
    
        // Ensure the object is initialized (in case init() is needed)
        if (method_exists($shipping_method, 'init')) {
            $shipping_method->init();
        }
    
        // Clear transient first
        delete_transient('tipax_addresses');
    
        $addresses_objects = $shipping_method->loadAddresses();
    } catch (Throwable $e) {
        echo '<div class="notice notice-error"><p>خطای داخلی در دریافت آدرس‌ها: ' . esc_html($e->getMessage()) . '</p></div>';
        wp_die();
    }

    if (!$addresses_objects || empty($addresses_objects->address)) {
        echo '<div class="notice notice-warning"><p>هیچ آدرس پیش‌فرضی یافت نشد. لطفاً در پنل ای‌تیپاکس، یک آدرس ایجاد و آن را پیش‌فرض کنید.</p></div>';
        wp_die();
    }

    $addresses = is_array($addresses_objects->address) ? $addresses_objects->address : [$addresses_objects->address];

    echo '<table class="widefat striped">';
    echo '<thead><tr><th>آدرس کامل</th></tr></thead><tbody>';

    foreach ($addresses as $address) {
        echo '<tr><td>' . esc_html($address->fullAddress) . '</td></tr>';
    }

    echo '</tbody></table>';

    wp_die();
}

function tipax_logs_page() {
    tipax_admin_tabs('tipax-logs');
    if (!tipax_is_connection_validated()) {
        echo '<div class="notice notice-error"><p>برای دسترسی به این بخش ابتدا باید <a href="' . admin_url('admin.php?page=tipax-api-config') . '">تنظیمات اتصال</a> را کامل کرده و اتصال را <a href="' . admin_url('admin.php?page=tipax-test-connection') . '">آزمایش</a> کنید.</p></div>';
        return;
    }

    if (get_option('tipax_has_address') !== 'yes') {
        echo '<div class="notice notice-warning"><p>برای استفاده از این بخش باید حداقل یک آدرس ثبت شده در حساب تیپاکس شما باشد. لطفاً به <a href="' . admin_url('admin.php?page=tipax-addresses') . '">بخش آدرس‌ها</a> مراجعه کنید.</p></div>';
        return;
    }

    $log_path = plugin_dir_path(__FILE__) . '../../logs/tipax_shipping.log';

    echo '<h2>Tipax Log File</h2>';
    echo '<p>در این بخش می‌توانید لاگ‌های ثبت شده توسط افزونه تیپاکس را مشاهده یا پاک کنید.</p>';

    echo '<button id="clear_tipax_log" class="button button-secondary">🧹 پاک‌سازی لاگ</button>';
    echo '<pre id="tipax_log_output" style="background:#fff; padding:20px; margin-top:20px; max-height:600px; overflow:auto; border:1px solid #ccd0d4;direction:ltr">در حال بارگذاری...</pre>';

    ?>
    <script>
        jQuery(document).ready(function($) {
            function loadLog() {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'tipax_load_log'
                    },
                    success: function(response) {
                        $('#tipax_log_output').text(response);
                    },
                    error: function() {
                        $('#tipax_log_output').text('خطا در بارگیری لاگ.');
                    }
                });
            }

            $('#clear_tipax_log').on('click', function() {
                if (!confirm('آیا مطمئن هستید که می‌خواهید لاگ را پاک کنید؟')) return;

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'tipax_clear_log'
                    },
                    success: function(response) {
                        $('#tipax_log_output').text('');
                    },
                    error: function() {
                        $('#tipax_log_output').text('خطا در پاک کردن لاگ.');
                    }
                });
            });

            loadLog();
        });
    </script>
    <?php
}


add_action('wp_ajax_tipax_load_log', function() {
    $log_path = plugin_dir_path(__FILE__) . '../logs/tipax_shipping.log';
    if (!file_exists($log_path)) {
        echo "فایلی با این مسیر یافت نشد.";
        wp_die();
    }

    $contents = file_get_contents($log_path);
    echo esc_textarea($contents ?: 'فایل لاگ خالی است.');
    wp_die();
});

add_action('wp_ajax_tipax_clear_log', function() {
    $log_path = plugin_dir_path(__FILE__) . '../logs/tipax_shipping.log';

    if (file_exists($log_path)) {
        file_put_contents($log_path, '');
        echo 'Log cleared.';
    } else {
        echo 'Log file not found.';
    }

    wp_die();
});

function tipax_orders_page() {
    tipax_admin_tabs('tipax-orders');

    echo '<h2>سفارشات تیپاکس</h2>';

    $paged    = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
    $per_page = 10;
    $after = date('Y-m-d H:i:s', strtotime('-10 days'));

    $args = [
        'status'     => ['processing', 'completed', 'on-hold', 'pending'],
        'limit'      => $per_page,
        'paged'      => $paged,
        'orderby'    => 'date',
        'order'      => 'DESC',
        'date_query' => [
            [
                'after'     => $after,
                'inclusive' => true,
            ],
        ],
        'shipping_method' => 'tipax_shipping_method',
    ];

    $query = new WC_Order_Query($args);
    $tipax_orders = $query->get_orders();

    $args['limit'] = -1;
    $args['paged'] = '';
    $count_query   = new WC_Order_Query($args);
    $total_orders  = count($count_query->get_orders());
    $total_pages   = ceil($total_orders / $per_page);

    echo '<table class="widefat striped"><thead><tr>
        <th>شماره سفارش</th>
        <th>تاریخ</th>
        <th>نام مشتری</th>
        <th>محصولات</th>
        <th>کد پیگیری</th>
        <th>کد قرارداد</th>
        <th>وضعیت</th>
        <th>اقدامات</th>
    </tr></thead><tbody>';

    if (empty($tipax_orders)) {
        echo '<tr><td colspan="8">سفارشی یافت نشد.</td></tr>';
    }

    foreach ($tipax_orders as $order) {
        $first_item    = current($order->get_items());
        $tracking_code = $first_item ? $first_item->get_meta('tipax_tracking_code') : '';
        $saved_status  = get_post_meta($order->get_id(), '_tipax_saved_status', true);
        $saved_contract= get_post_meta($order->get_id(), '_tipax_saved_contract', true);
        
        $billing_name  = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        
        $product_titles = [];
        foreach ($order->get_items() as $item) {
            $product_titles[] = $item->get_name() . ' × ' . $item->get_quantity();
        }
        $products_string = implode(', ', $product_titles);

        echo '<tr>';
        echo '<td>#' . $order->get_id() . '</td>';
        echo '<td>' . esc_html($order->get_date_created()->date_i18n('Y-m-d H:i')) . '</td>'; // ✅ show order date
        echo '<td>' . esc_html($billing_name) . '</td>';
        echo '<td>' . esc_html($products_string) . '</td>';
        echo '<td>' . esc_html($tracking_code) . '</td>';
        echo '<td>' . esc_html($saved_contract) . '</td>';
        echo '<td>' . esc_html($saved_status ?: 'نامشخص') . '</td>';
        echo '<td>';
        if (empty($tracking_code)) {
            echo '<button class="button tipax-register-order" data-order-id="' . esc_attr($order->get_id()) . '">ثبت مجدد در تیپاکس</button>';
        } else {
            echo '<button class="button tipax-update-status" data-order-id="' . esc_attr($order->get_id()) . '">بروزرسانی وضعیت</button>';
        }
        echo '</td>';
        echo '</tr>';
    }

    echo '</tbody></table>';

    // ✅ Pagination buttons
    echo '<div style="margin-top:20px;">';
    for ($i = 1; $i <= $total_pages; $i++) {
        echo '<a href="?page=tipax-orders&paged=' . $i . '" class="button' . ($i === $paged ? ' button-primary' : '') . '">' . $i . '</a> ';
    }
    echo '</div>';

    // JS part unchanged...
    ?>
    <script>
    jQuery(document).ready(function($) {
        $('.tipax-update-status').click(function(e) {
            e.preventDefault();
            let button = $(this);
            let orderId = button.data('order-id');

            button.prop('disabled', true).text('در حال بروزرسانی...');

            $.post(ajaxurl, {
                action: 'tipax_update_order_status',
                order_id: orderId,
                _ajax_nonce: '<?php echo wp_create_nonce('tipax_update_order'); ?>'
            }, function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    button.text('خطا').prop('disabled', false);
                    alert(response.data);
                }
            });
        });

        $('.tipax-register-order').click(function(e) {
            e.preventDefault();
            let button = $(this);
            let orderId = button.data('order-id');

            button.prop('disabled', true).text('در حال ثبت...');

            $.post(ajaxurl, {
                action: 'tipax_register_order',
                order_id: orderId,
                _ajax_nonce: '<?php echo wp_create_nonce('tipax_register_order'); ?>'
            }, function(response) {
                if (response.success) {
                    button.text('ثبت شد');
                    location.reload();
                } else {
                    button.text('خطا').prop('disabled', false);
                    alert(response.data);
                }
            });
        });
    });
    </script>
    <?php
}




function tipax_update_order_status($order_id) {
    $order = wc_get_order($order_id);
    if (!$order) return;

    $item = current($order->get_items());
    $tracking_code = $item->get_meta('tipax_tracking_code');

    if (!$tracking_code) {
        echo '<div class="notice notice-error"><p>کد پیگیری برای این سفارش یافت نشد.</p></div>';
        return;
    }

    $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken');
    $request = new Request();

    // 🟡 1. First request by tracking code
    $response = $request->apiRequest('Tracking/ByTrackingCode', [$tracking_code], 'POST', $accessToken);

    if (isset($response[0]->status)) {
        $status = $response[0]->status;
        $contract = $response[0]->priceDetail->jetContractCode ?? '';

        update_post_meta($order_id, '_tipax_saved_status', $status);
        update_post_meta($order_id, '_tipax_saved_contract', $contract);

        echo '<div class="notice notice-success"><p>وضعیت سفارش بروزرسانی شد: ' . esc_html($status) . '</p></div>';
    } else {
        echo '<div class="notice notice-error"><p>دریافت وضعیت سفارش از تیپاکس با خطا مواجه شد.</p></div>';
        return;
    }

    // 🟡 2. Optional: Track by contract code
    if (!empty($contract)) {
        $contractRes = $request->apiRequest("Tracking/ByContractCode/$contract", [], 'GET', $accessToken);
        // Optional: Save full contract data in post meta
        update_post_meta($order_id, '_tipax_contract_tracking_data', $contractRes);
    }
}

add_action('wp_ajax_tipax_update_order_status', 'tipax_ajax_update_order_status');

function tipax_ajax_update_order_status() {
    check_ajax_referer('tipax_update_order');

    $order_id = absint($_POST['order_id'] ?? 0);
    if (!$order_id) {
        wp_send_json_error('شناسه سفارش نامعتبر است.');
    }

    $order = wc_get_order($order_id);
    if (!$order) {
        wp_send_json_error('سفارش یافت نشد.');
    }

    $item = current($order->get_items());
    $tracking_code = explode(" ", strtolower($item->get_meta('tipax_tracking_code')));

    if (!$tracking_code[0]) {
        wp_send_json_error('کد پیگیری موجود نیست.');
    }

    $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken');
    $request = new Request();

    $response = $request->apiRequest('Tracking/ByTrackingCode', [$tracking_code[0]], 'POST', $accessToken);
    
    if (!isset($response[0]->status)) {
        wp_send_json_error('پاسخ نامعتبر از تیپاکس.');
    }

    $status = $response[0]->status;
    $contract = $response[0]->priceDetail->jetContractCode ?? '';

    update_post_meta($order_id, '_tipax_saved_status', $status);
    update_post_meta($order_id, '_tipax_saved_contract', $contract);

    if (!empty($contract)) {
        $contractRes = $request->apiRequest("Tracking/ByContractCode/$contract", [], 'GET', $accessToken);
        update_post_meta($order_id, '_tipax_contract_tracking_data', $contractRes);
    }

    wp_send_json_success([
        'status' => $status,
        'contract' => $contract,
    ]);
}
add_action('wp_ajax_tipax_register_order', 'tipax_ajax_register_order');

function tipax_ajax_register_order() {
    check_ajax_referer('tipax_register_order');

    $order_id = absint($_POST['order_id'] ?? 0);
    if (!$order_id) {
        wp_send_json_error('شناسه سفارش نامعتبر است.');
    }

    $order = wc_get_order($order_id);
    if (!$order || !$order->has_shipping_method('tipax_shipping_method')) {
        wp_send_json_error('سفارش معتبر نیست یا مربوط به تیپاکس نیست.');
    }

    unset($_COOKIE["tipaxOrder_".$order_id]);
    $result = get_tipax_tracking_code_after_order_failed($order_id);

    if ($result['success']) {
        wp_send_json_success(implode("\n", $result['messages']));
    } else {
        wp_send_json_error(implode("\n", $result['messages']));
    }
}


