<?php
/**
 * Tipax Order Confirmation — Aggregated Packages (≤200cm per side, ≤100kg per package)
 * Endpoint unchanged: Orders/WithPreDefinedOrigin
 *
 * PHP 7.4+ compatible (no arrow functions, no typed properties).
 */

/**
 * Return TRUE if the origin (by addressesId) city equals destinationCityId, else FALSE.
 * If $originAddressesId is null/0, it will use the default address (isDefault === true).
 */
function tipax_same_city_bool($request, $accessToken, $originAddressesId, $destinationCityId): bool
{
    $resp = $request->apiRequest('Addresses', [], 'GET', $accessToken);
    if (!is_array($resp)) {
        return false;
    }

    // Index by addressesId (NOT by id)
    $byAddressesId = [];
    $default = null;
    foreach ($resp as $addr) {
        if (isset($addr->addressesId)) {
            $byAddressesId[(int)$addr->addressesId] = $addr;
        }
        if (!empty($addr->isDefault)) {
            $default = $addr;
        }
    }

    // Fallback to default if origin not provided
    if (empty($originAddressesId) && $default) {
        $originAddressesId = (int)$default->addressesId;
    }

    if (!isset($byAddressesId[(int)$originAddressesId])) {
        return false;
    }

    $origin = $byAddressesId[(int)$originAddressesId];
    $originCity = $origin->address->cityETipaxId ?? null;
    return (isset($originCity) && (int)$originCity === (int)$destinationCityId);
}

/** Helpers **/
function tipax_pick_value($productVal, $settingsVal, $absoluteDefault)
{
    if ($productVal !== null && $productVal > 0) return (float)$productVal;
    if ($settingsVal !== null && $settingsVal > 0) return (float)$settingsVal;
    return (float)$absoluteDefault;
}

function tipax_to_kg($val, $weight_unit)
{
    $v = (float)$val;
    if ($v <= 0) return 0.0;
    $u = strtolower((string)$weight_unit);
    switch ($u) {
        case 'kg': return $v;
        case 'g':  return $v / 1000.0;
        case 'lbs':
        case 'lb': return $v * 0.45359237;
        case 'oz': return $v * 0.028349523125;
        default:   return $v; // assume kg
    }
}

function tipax_dim_ratio_to_cm($dimension_unit)
{
    switch (strtolower((string)$dimension_unit)) {
        case 'm':  return 100.0;  // meters → cm
        case 'cm': return 1.0;    // cm → cm
        case 'mm': return 0.1;    // mm → cm
        case 'in': return 2.54;   // inch → cm
        case 'yd': return 91.44;  // yard → cm
        default:   return 1.0;
    }
}

/**
 * Build packages[] payload for Orders/WithPreDefinedOrigin
 * Uses same aggregation logic as pricing: minimal number of packages such that
 * each package has side ≤ 200cm and weight ≤ 100kg.
 *
 * Returns array like:
 * [
 *   'packages' => [ stdClass, stdClass, ... ],
 *   'serviceId' => int
 * ]
 */
function tipax_build_packages_for_order(
    WC_Order $order,
    $request,
    $accessToken,
    $shppingAddress,               // addressesId (origin)
    $order_shipping_city_id,
    $order_shipping_address_1,
    $order_shipping_postcode,
    $order_receiver_plaque,
    $order_shipping_first_name,
    $order_shipping_last_name,
    $order_receiver_phone,
    $order_billing_phone,
    $woocommerce_currency,
    $weight_unit,
    $dimension_unit,
    $avgWeight,
    $avgLength,
    $avgWidth,
    $avgHeight,
    $checkout_method
) {
    // Decide serviceId (same-city → 7, otherwise 5)
    $serviceId = tipax_same_city_bool($request, $accessToken, $shppingAddress, $order_shipping_city_id) ? 7 : 5;

    // Units
    $unit_ratio = tipax_dim_ratio_to_cm($dimension_unit);

    // Totals
    $total_item_qty        = 0;
    $total_weight_kg       = 0.0;
    $total_volume_cm3      = 0.0;
    $items_total_store     = 0.0;

    // Single item snapshot (for precise single-item splitting)
    $single_item_snapshot = null;

    foreach ($order->get_items() as $li) {
        $items_total_store += (float)$li->get_total();
    }

    foreach ($order->get_items() as $it) {
        $qty     = (int)$it->get_quantity();
        if ($qty <= 0) continue;

        $product = $it->get_product();
        if (!$product) continue;

        $rawW = is_numeric($product->get_weight()) ? (float)$product->get_weight() : null;
        $rawL = is_numeric($product->get_length()) ? (float)$product->get_length() : null;
        $rawH = is_numeric($product->get_height()) ? (float)$product->get_height() : null;
        $rawD = is_numeric($product->get_width())  ? (float)$product->get_width()  : null;

        $wKg_logic = tipax_pick_value($rawW, $avgWeight, 6.0);  // logical kg
        $L_logic   = tipax_pick_value($rawL, $avgLength, 45.0); // logical cm
        $H_logic   = tipax_pick_value($rawH, $avgHeight, 45.0); // logical cm
        $W_logic   = tipax_pick_value($rawD, $avgWidth,  45.0); // logical cm

        $item_weight_kg = tipax_to_kg($wKg_logic, $weight_unit);
        $L_cm = $L_logic * $unit_ratio;
        $H_cm = $H_logic * $unit_ratio;
        $W_cm = $W_logic * $unit_ratio;

        if ($item_weight_kg <= 0 || $L_cm <= 0 || $W_cm <= 0 || $H_cm <= 0) {
            tipax_log("Tipax order: invalid product dimensions/weight detected.");
            return ['packages' => [], 'serviceId' => $serviceId];
        }

        $total_item_qty   += $qty;
        $total_weight_kg  += $item_weight_kg * $qty;
        $total_volume_cm3 += ($L_cm * $W_cm * $H_cm) * $qty;

        if ($total_item_qty === $qty && $qty === 1) {
            $single_item_snapshot = [
                'weight_kg' => $item_weight_kg,
                'L_cm'      => $L_cm,
                'W_cm'      => $W_cm,
                'H_cm'      => $H_cm,
                'price'     => (float)$product->get_price(),
            ];
        }
    }

    // thresholds for "light"
    $threshold_store = 4000000.0; // Toman
    if (strtoupper($woocommerce_currency) === 'IRR') $threshold_store *= 10;

    // constants
    $MAX_DIM_CM      = 200.0;
    $PER_ITEM_MAX_KG = 100.0;
    $CUBE_CM3        = $MAX_DIM_CM * $MAX_DIM_CM * $MAX_DIM_CM;

    // Build packages[]
    $packages = [];

    // Helper to make one package object (destination/receiver/values + meta)
    $make_pkg = function ($weight_kg, $side_cm, $value_store) use (
        $shppingAddress,
        $order_shipping_city_id,
        $order_shipping_address_1,
        $order_shipping_postcode,
        $order_receiver_plaque,
        $order_shipping_first_name,
        $order_shipping_last_name,
        $order_receiver_phone,
        $order_billing_phone,
        $woocommerce_currency,
        $checkout_method
    ) {
        $pkg = new stdClass;
        $pkg->originId = $shppingAddress;

        $pkg->destination = new stdClass;
        $pkg->destination->cityId      = $order_shipping_city_id;
        $pkg->destination->districtId  = 0;
        $pkg->destination->fullAddress = $order_shipping_address_1 ?? null;
        $pkg->destination->unit        = null;
        $pkg->destination->no          = $order_receiver_plaque;
        $pkg->destination->floor       = null;
        $pkg->destination->postalCode  = $order_shipping_postcode;

        $pkg->destination->beneficiary = new stdClass;
        $pkg->destination->beneficiary->fullName = $order_shipping_first_name . ' ' . $order_shipping_last_name;
        $pkg->destination->beneficiary->mobile   = $order_receiver_phone;
        $pkg->destination->beneficiary->phone    = $order_billing_phone;

        $pkg->receiver = $pkg->destination->beneficiary;

        $pkg->weight = (float)$weight_kg;
        $pkg->length = (float)$side_cm;
        $pkg->width  = (float)$side_cm;
        $pkg->height = (float)$side_cm;

        $pkg->packageValue = (strtoupper($woocommerce_currency) === 'IRT')
            ? (float)($value_store * 10.0)
            : (float)$value_store;

        // Determine packing (minipack if tiny/light else box)
        if ($pkg->weight > 0 && $pkg->weight < 2.0 && $pkg->length <= 10 && $pkg->width <= 10 && $pkg->height <= 5) {
            $pkg->packingId        = 512;
            $pkg->packageContentId = 10;
            $pkg->packType         = 50;
        } else {
            $pkg->packingId        = 504;
            $pkg->packageContentId = 9;
            $pkg->packType         = 20;
        }

        $pkg->description         = "";
        $pkg->enableLabelPrivacy  = true;  // always true (confirmed)
        $pkg->paymentType         = $checkout_method;
        $pkg->pickupType          = 10;
        $pkg->distributionType    = 10;
        $pkg->parcelBookId        = 0;
        $pkg->isUnusual           = false;

        return $pkg;
    };

    // Single item precise mode (keep item dimensions feel), but still split if limits exceeded
    if ($total_item_qty === 1 && $single_item_snapshot) {
        $w_kg    = $single_item_snapshot['weight_kg'];
        $L       = $single_item_snapshot['L_cm'];
        $W       = $single_item_snapshot['W_cm'];
        $H       = $single_item_snapshot['H_cm'];
        $maxSide = max($L, $W, $H);
        $vol     = $L * $W * $H;
        $price   = 0.0;

        // use order items_total_store as value for single item if price missing
        if (isset($single_item_snapshot['price'])) {
            $price = (float)$single_item_snapshot['price'];
        } else {
            $price = 0.0;
            foreach ($order->get_items() as $li) $price += (float)$li->get_total();
        }

        $n_by_kg  = (int)ceil($w_kg / $PER_ITEM_MAX_KG);
        $n_by_dim = (int)ceil(($maxSide > 0 ? $maxSide / $MAX_DIM_CM : 0));
        $n_by_vol = (int)ceil($vol / $CUBE_CM3);

        $num_pkgs = max(1, max($n_by_kg, max($n_by_dim, $n_by_vol)));

        $pi_weight = $w_kg / $num_pkgs;
        $pi_value  = $price / $num_pkgs;
        $remaining_vol = $vol;

        for ($i = 1; $i <= $num_pkgs; $i++) {
            if ($i < $num_pkgs) {
                $side      = $MAX_DIM_CM;
                $pkg_value = $pi_value;
                $pkg_w     = $pi_weight;
                $remaining_vol -= $CUBE_CM3;
                if ($remaining_vol < 0) $remaining_vol = 0;
            } else {
                $side = ($remaining_vol > 0) ? pow($remaining_vol, 1/3) : $MAX_DIM_CM;
                if ($side > $MAX_DIM_CM) $side = $MAX_DIM_CM;
                $pkg_value = $pi_value;
                $pkg_w     = $pi_weight;
            }

            $pkg = $make_pkg($pkg_w, ceil($side), $pkg_value);
            $packages[] = $pkg;
        }
    } else {
        // Aggregate multi-item / multi-qty into minimal number of packages
        if ($total_weight_kg <= 0 || $total_volume_cm3 <= 0) {
            tipax_log("Tipax order: invalid cart totals.");
            return ['packages' => [], 'serviceId' => $serviceId];
        }

        $number_by_kg  = (int)ceil($total_weight_kg / $PER_ITEM_MAX_KG);
        $number_by_vol = (int)ceil($total_volume_cm3 / $CUBE_CM3);
        $package_count = max(1, max($number_by_kg, $number_by_vol));

        if ($total_item_qty > 0) {
            $package_count = min($package_count, $total_item_qty);
        }

        $pi_weight = $total_weight_kg  / $package_count;
        $pi_value  = $items_total_store / $package_count;

        $remaining_vol = $total_volume_cm3;

        for ($i = 1; $i <= $package_count; $i++) {
            if ($i < $package_count) {
                $side      = $MAX_DIM_CM;
                $pkg_w     = $pi_weight;
                $pkg_value = $pi_value;
                $remaining_vol -= $CUBE_CM3;
                if ($remaining_vol < 0) $remaining_vol = 0;
            } else {
                $side = ($remaining_vol > 0) ? pow($remaining_vol, 1/3) : $MAX_DIM_CM;
                if ($side > $MAX_DIM_CM) $side = $MAX_DIM_CM;
                $pkg_w     = $pi_weight;
                $pkg_value = $pi_value;
            }

            // Guard: if floating divisions caused weight overflow > 100kg, split finer
            if ($pkg_w > $PER_ITEM_MAX_KG) {
                $parts = (int)ceil($pkg_w / $PER_ITEM_MAX_KG);
                $w_each = $pkg_w / $parts;
                $v_each = $pkg_value / $parts;
                for ($k = 0; $k < $parts; $k++) {
                    $packages[] = $make_pkg($w_each, ceil($side), $v_each);
                }
            } else {
                $packages[] = $make_pkg($pkg_w, ceil($side), $pkg_value);
            }
        }
    }

    // Apply serviceId to all packages (same as pricing rule)
    foreach ($packages as $pkg) {
        $pkg->serviceId = $serviceId;
    }

    return ['packages' => $packages, 'serviceId' => $serviceId];
}

/**
 * MAIN: Submit order to Tipax on thank-you page (success path)
 */
add_action('woocommerce_before_thankyou', 'get_tipax_tracking_code', 2000, 1);
function get_tipax_tracking_code($order_id)
{
    $request = new Request();
    $order   = wc_get_order($order_id);
    if (!$order || !$order->has_shipping_method('tipax_shipping_method')) return;

    $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);

    $order_data = $order->get_data();
    $orderShippingGetItems = $order->get_items('shipping');

    // Extract serviceId saved in shipping item meta (used for label text decisions elsewhere)
    $serviceId = null;
    if (!empty($orderShippingGetItems)) {
        $firstShip = reset($orderShippingGetItems);
        foreach ($firstShip->get_meta_data() as $item) {
            $serviceId = $item->value;
        }
    }

    // Receiver/Address
    $order_receiver_phone   = $order_data['billing']['phone'];
    $order_receiver_plaque  = null;
    foreach ($order_data["meta_data"] as $meta) {
        if ($meta->key == "phone")  $order_receiver_phone  = $meta->value;
        if ($meta->key == "plaque") $order_receiver_plaque = $meta->value;
    }

    $order_shipping_first_name = $order_data['shipping']['first_name'];
    $order_shipping_last_name  = $order_data['shipping']['last_name'];
    $order_billing_phone       = $order_data['billing']['phone'];
    $order_shipping_postcode   = postalcodeValidate($order_data['shipping']['postcode']);
    $order_shipping_address_1  = $order_data['shipping']['address_1'];
    $order_shipping_city       = $order->get_shipping_city();
    $order_shipping_country    = $order->get_shipping_country();
    $order_shipping_state      = $order->get_shipping_state();

    $states = WC()->countries->get_states($order_shipping_country);
    $stateTextShipping = isset($states[$order_shipping_state]) ? $states[$order_shipping_state] : $order_shipping_state;

    // Store info
    $storeRawCountry   = get_option('woocommerce_default_country'); // e.g. IR:THR
    $weight_unit       = get_option('woocommerce_weight_unit');
    $dimension_unit    = get_option('woocommerce_dimension_unit');
    $woocommerce_currency = get_option('woocommerce_currency');

    // Country/state split
    $parts = explode(':', (string)$storeRawCountry);
    $storeCountry = $parts[0] ?? '';
    $storeState   = $parts[1] ?? '';
    $storeCity    = get_option('woocommerce_store_city');

    $storeStateReal = WC()->countries->get_states($storeCountry)[$storeState] ?? $storeState;

    // City codes
    $order_shipping_city_id = get_city_code($order_shipping_city, $stateTextShipping);
    $origin_city_id         = get_city_code($storeCity, $storeStateReal);

    // Settings
    $shipping_instance_id = null;
    foreach ($orderShippingGetItems as $item_id => $item) {
        $shipping_instance_id = $item->get_instance_id();
        break;
    }
    $settings = get_option("woocommerce_tipax_shipping_method_" . $shipping_instance_id . "_settings", null);
    if ($settings == null) {
        $settings = get_option("woocommerce_tipax_shipping_method_settings", null);
    }

    $addressMethod = 3; // confirmed
    $checkout_method = 10;
    $collection_time = 1;
    $avgWeight = $avgLength = $avgWidth = $avgHeight = null;

    if ($settings != null) {
        $shppingAddress   = get_option("tipax_shipping_address"); // addressesId
        $plugin_settings  = get_option('woocommerce_tipax_shipping_method_settings', []);
        $checkout_method  = $plugin_settings['checkout_method'] ?? 10;
        if ($checkout_method == 100) $checkout_method = 50;
        $collection_time  = $plugin_settings['collection_time'] ?? 1;

        $avgWeight = (isset($plugin_settings['avg_weight']) && is_numeric($plugin_settings['avg_weight'])) ? (float)$plugin_settings['avg_weight'] : null;
        $avgLength = (isset($plugin_settings['avg_length']) && is_numeric($plugin_settings['avg_length'])) ? (float)$plugin_settings['avg_length'] : null;
        $avgWidth  = (isset($plugin_settings['avg_width'])  && is_numeric($plugin_settings['avg_width']))  ? (float)$plugin_settings['avg_width']  : null;
        $avgHeight = (isset($plugin_settings['avg_height']) && is_numeric($plugin_settings['avg_height'])) ? (float)$plugin_settings['avg_height'] : null;
    } else {
        $shppingAddress = null;
        $plugin_settings  = get_option('woocommerce_tipax_shipping_method_settings', []);
        $checkout_method  = $plugin_settings['checkout_method'] ?? 10;
        if ($checkout_method == 100) $checkout_method = 50;
        $collection_time  = $plugin_settings['collection_time'] ?? 1;
    }

    // Required params
    if ($addressMethod != 3 || empty($order_shipping_city_id) || empty($shppingAddress)) {
        return; // cannot proceed
    }

    // Build packages with aggregation logic
    $build = tipax_build_packages_for_order(
        $order, $request, $accessToken,
        $shppingAddress,
        $order_shipping_city_id,
        $order_shipping_address_1,
        $order_shipping_postcode,
        $order_receiver_plaque,
        $order_shipping_first_name,
        $order_shipping_last_name,
        $order_receiver_phone,
        $order_billing_phone,
        $woocommerce_currency,
        $weight_unit,
        $dimension_unit,
        $avgWeight,
        $avgLength,
        $avgWidth,
        $avgHeight,
        $checkout_method
    );

    if (empty($build['packages'])) {
        wc_print_notice(__('There was a problem building Tipax packages.', 'delivery-service-tipax-shipping'), 'error');
        return;
    }

    // Apply serviceId (already applied inside builder, but keep compatible with your saved meta)
    if (!empty($serviceId)) {
        foreach ($build['packages'] as $p) { $p->serviceId = (int)$serviceId; }
    }

    $data = new stdClass;
    $data->packages = $build['packages'];

    // Submit
    storeTipaxOrder("Orders/WithPreDefinedOrigin", $order, $request, $data, $accessToken, $order_id);
}

/**
 * Submit to API and persist meta/cookie (SUCCESS path)
 */
function storeTipaxOrder($apiPath, $order, Request $request, stdClass $data, $accessToken, $order_id)
{
    foreach ($order->get_items() as $item) { $firstItem = $item; break; }

    echo '<ul>';

    $tipax_code = isset($firstItem)
        ? ($firstItem->get_meta("tipax_tracking_code") . $firstItem->get_meta("tipax_order_id"))
        : '';

    if (!$tipax_code) {
        // anti-dup cookie
        $tipaxOrderCookie = tipaxPluginCookie("tipaxOrder_" . $order_id);
if ($tipaxOrderCookie) {
    $tipaxOrderInfo = json_decode(base64_decode($tipaxOrderCookie), false);
    $trackingCodes  = isset($tipaxOrderInfo->trackingCodes) && is_array($tipaxOrderInfo->trackingCodes)
        ? array_map('strtoupper', $tipaxOrderInfo->trackingCodes)
        : [];
    $orderId        = $tipaxOrderInfo->orderId ?? null;

    $totalCodes = count($trackingCodes);
    $i = 0;

    foreach ($order->get_items() as $item_id => $item) {
        $product   = $item->get_product();
        $quantity  = (int)$item->get_quantity();
        $assigned  = [];

        if ($totalCodes <= 1) {
            // Only one tracking code returned → assign same code to all items
            $assigned = $trackingCodes ? [$trackingCodes[0]] : [];
        } else {
            // Distribute sequentially
            for ($j = 0; $j < $quantity; $j++) {
                if ($i < $totalCodes && !empty($trackingCodes[$i])) {
                    $assigned[] = $trackingCodes[$i];
                    $i++;
                }
            }
        }

        $codes_str = implode(' ', $assigned);
        if ($codes_str !== '') {
            wc_add_order_item_meta($item_id, 'tipax_tracking_code', $codes_str);
        }
        if ($orderId) {
            wc_add_order_item_meta($item_id, 'tipax_order_id', $orderId);
        }

        echo '<li><strong>' . __('Tipax order information for ', 'delivery-service-tipax-shipping') . ' : ' . esc_html($product->get_name()) . '</strong></li>';
        echo '<li>' . __('Tipax Tracking Code', 'delivery-service-tipax-shipping') . ' : ' . esc_html($codes_str !== '' ? $codes_str : '-') . '</li>';
        echo '<li>' . __('Tipax Order Id', 'delivery-service-tipax-shipping') . ' : ' . esc_html($orderId) . '</li>';
        echo '<br>';
    }
} else {
            $response = $request->apiRequest($apiPath, $data, 'POST', $accessToken);

            if (isset($response->errors)) {
                wc_print_notice(__('There was a problem in creating tipax order', 'delivery-service-tipax-shipping'), 'error');
                wc_print_notice(__('Please contact us for fixing the error', 'delivery-service-tipax-shipping'), 'error');
                foreach ($response->errors as $error) {
                    if (!empty($error->errorMessage)) {
                        wc_print_notice($error->errorMessage, 'error');
                    }
                }
                echo '</ul>';
                return;
            }
            if (isset($response->status) && (int)$response->status == 401) {
                $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);
                $response = $request->apiRequest($apiPath, $data, 'POST', $accessToken);
            }

            // cache response for 7 days
            $expiration = current_time('timestamp') + (DAY_IN_SECONDS * 7);
            tipaxPluginCookie("tipaxOrder_" . $order_id, base64_encode(json_encode($response)), $expiration);

            if (isset($response->trackingCodes) && is_array($response->trackingCodes)) {
                wc_print_notice(__('Tipax order shipping created successfully', 'delivery-service-tipax-shipping'), 'notice');

                $i = 0;
$trackingCodes = isset($response->trackingCodes) && is_array($response->trackingCodes)
    ? array_map('strtoupper', $response->trackingCodes)
    : [];
$orderId = $response->orderId ?? null;

$totalCodes = count($trackingCodes);
$i = 0;

foreach ($order->get_items() as $item_id => $item) {
    $product  = $item->get_product();
    $quantity = (int)$item->get_quantity();
    $assigned = [];

    if ($totalCodes <= 1) {
        // Only one code → apply same one to all items
        $assigned = $trackingCodes ? [$trackingCodes[0]] : [];
    } else {
        for ($j = 0; $j < $quantity; $j++) {
            if ($i < $totalCodes && !empty($trackingCodes[$i])) {
                $assigned[] = $trackingCodes[$i];
                $i++;
            }
        }
    }

    $codes_str = implode(' ', $assigned);
    if ($codes_str !== '') {
        wc_add_order_item_meta($item_id, 'tipax_tracking_code', $codes_str);
    }
    if ($orderId) {
        wc_add_order_item_meta($item_id, 'tipax_order_id', $orderId);
    }

    echo '<li><strong>' . __('Tipax order information for ', 'delivery-service-tipax-shipping') . ' : ' . esc_html($product->get_name()) . '</strong></li>';
    echo '<li>' . __('Tipax Tracking Code', 'delivery-service-tipax-shipping') . ' : ' . esc_html($codes_str !== '' ? $codes_str : '-') . '</li>';
    echo '<li>' . __('Tipax Order Id', 'delivery-service-tipax-shipping') . ' : ' . esc_html($orderId) . '</li>';
    echo '<br>';
}

            }
        }
    } else {
wc_print_notice(__('Tipax order shipping created successfully', 'delivery-service-tipax-shipping'), 'notice');

$shown = [];
foreach ($order->get_items() as $item_id => $item) {
    $product  = $item->get_product();
    $codesRaw = trim((string)$item->get_meta("tipax_tracking_code"));
    $codesArr = array_filter(array_unique(preg_split('/\s+/', $codesRaw)));
    $codesStr = implode(' ', $codesArr);

    // Avoid duplicate lines if codes already shown
    if (in_array($codesStr, $shown, true)) {
        continue;
    }
    $shown[] = $codesStr;

    echo '<li><strong>' . __('Tipax order information for ', 'delivery-service-tipax-shipping') . ' : ' . esc_html($product->get_name()) . '</strong></li>';
    echo '<li>' . __('Tipax Tracking Code', 'delivery-service-tipax-shipping') . ' : ' . esc_html($codesStr !== '' ? $codesStr : '-') . '</li>';
    echo '<li>' . __('Tipax Order Id', 'delivery-service-tipax-shipping') . ' : ' . esc_html($item->get_meta("tipax_order_id")) . '</li>';
    echo '<br>';
}

    }

    echo '</ul>';
}

/**
 * FAILED/RETRY path: identical aggregation & persistence, returns normalized array.
 */
function get_tipax_tracking_code_after_order_failed($order_id)
{
    $request = new Request();
    $order   = wc_get_order($order_id);
    if (!$order || !$order->has_shipping_method('tipax_shipping_method')) {
        return [
            'success' => false,
            'messages' => ['Order not using Tipax shipping.'],
            'order_id' => $order_id,
            'tracking_codes' => [],
            'tipax_order_id' => null,
            'http_status' => null,
        ];
    }

    $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);

    $order_data = $order->get_data();
    $orderShippingGetItems = $order->get_items('shipping');

    // Extract serviceId saved in shipping meta
    $serviceId = null;
    
    if (!empty($orderShippingGetItems)) {
        $firstShip = reset($orderShippingGetItems);
        foreach ($firstShip->get_meta_data() as $item) {
            $serviceId = $item->value;
        }
    }

    // Receiver/Address
    $order_receiver_phone   = $order_data['billing']['phone'];
    $order_receiver_plaque  = null;
    foreach ($order_data["meta_data"] as $meta) {
        if ($meta->key == "phone")  $order_receiver_phone  = $meta->value;
        if ($meta->key == "plaque") $order_receiver_plaque = $meta->value;
    }

    $order_shipping_first_name = $order_data['shipping']['first_name'];
    $order_shipping_last_name  = $order_data['shipping']['last_name'];
    $order_billing_phone       = $order_data['billing']['phone'];
    $order_shipping_postcode   = $order_data['shipping']['postcode'];
    $order_shipping_address_1  = $order_data['shipping']['address_1'];
    $order_shipping_city       = $order->get_shipping_city();
    $order_shipping_country    = $order->get_shipping_country();
    $order_shipping_state      = $order->get_shipping_state();

    $states = WC()->countries->get_states($order_shipping_country);
    $stateTextShipping = isset($states[$order_shipping_state]) ? $states[$order_shipping_state] : $order_shipping_state;

    // Store info
    $storeRawCountry   = get_option('woocommerce_default_country');
    $weight_unit       = get_option('woocommerce_weight_unit');
    $dimension_unit    = get_option('woocommerce_dimension_unit');
    $woocommerce_currency = get_option('woocommerce_currency');

    // Country/state split
    $parts = explode(':', (string)$storeRawCountry);
    $storeCountry = $parts[0] ?? '';
    $storeState   = $parts[1] ?? '';
    $storeCity    = get_option('woocommerce_store_city');

    $storeStateReal = WC()->countries->get_states($storeCountry)[$storeState] ?? $storeState;

    // City codes
    $order_shipping_city_id = get_city_code($order_shipping_city, $stateTextShipping);

    // Settings
    $shipping_instance_id = null;
    foreach ($orderShippingGetItems as $item_id => $item) { $shipping_instance_id = $item->get_instance_id(); break; }
    $settings = get_option("woocommerce_tipax_shipping_method_" . $shipping_instance_id . "_settings", null);
    if ($settings == null) {
        $settings = get_option("woocommerce_tipax_shipping_method_settings", null);
    }

    $addressMethod = 3;
    $checkout_method = 10;
    $collection_time = 1;
    $avgWeight = $avgLength = $avgWidth = $avgHeight = null;

    if ($settings != null) {
        $shppingAddress   = get_option("tipax_shipping_address"); // addressesId
        $plugin_settings  = get_option('woocommerce_tipax_shipping_method_settings', []);
        $checkout_method  = $plugin_settings['checkout_method'] ?? 10;
        if ($checkout_method == 100) $checkout_method = 50;
        $collection_time  = $plugin_settings['collection_time'] ?? 1;

        $avgWeight = (isset($plugin_settings['avg_weight']) && is_numeric($plugin_settings['avg_weight'])) ? (float)$plugin_settings['avg_weight'] : null;
        $avgLength = (isset($plugin_settings['avg_length']) && is_numeric($plugin_settings['avg_length'])) ? (float)$plugin_settings['avg_length'] : null;
        $avgWidth  = (isset($plugin_settings['avg_width'])  && is_numeric($plugin_settings['avg_width']))  ? (float)$plugin_settings['avg_width']  : null;
        $avgHeight = (isset($plugin_settings['avg_height']) && is_numeric($plugin_settings['avg_height'])) ? (float)$plugin_settings['avg_height'] : null;
    } else {
        $shppingAddress = null;
        $plugin_settings  = get_option('woocommerce_tipax_shipping_method_settings', []);
        $checkout_method  = $plugin_settings['checkout_method'] ?? 10;
        if ($checkout_method == 100) $checkout_method = 50;
        $collection_time  = $plugin_settings['collection_time'] ?? 1;
    }

    // Required params
    if ($addressMethod != 3 || empty($order_shipping_city_id) || empty($shppingAddress)) {
        return [
            'success' => false,
            'messages' => ['پارامترهای آدرس/شهر نامعتبر است.'],
            'order_id' => $order_id,
            'tracking_codes' => [],
            'tipax_order_id' => null,
            'http_status' => null,
        ];
    }

    // Build packages
    $build = tipax_build_packages_for_order(
        $order, $request, $accessToken,
        $shppingAddress,
        $order_shipping_city_id,
        $order_shipping_address_1,
        $order_shipping_postcode,
        $order_receiver_plaque,
        $order_shipping_first_name,
        $order_shipping_last_name,
        $order_receiver_phone,
        $order_billing_phone,
        $woocommerce_currency,
        $weight_unit,
        $dimension_unit,
        $avgWeight,
        $avgLength,
        $avgWidth,
        $avgHeight,
        $checkout_method
    );

    if (empty($build['packages'])) {
        return [
            'success' => false,
            'messages' => ['ساخت پکیج‌ها با خطا مواجه شد.'],
            'order_id' => $order_id,
            'tracking_codes' => [],
            'tipax_order_id' => null,
            'http_status' => null,
        ];
    }

    // Apply serviceId from shipping meta if present
    if (!empty($serviceId)) {
        foreach ($build['packages'] as $p) { $p->serviceId = (int)$serviceId; }
    }

    $data = new stdClass;
    $data->packages = $build['packages'];

    return storeTipaxOrderAfterFailed("Orders/WithPreDefinedOrigin", $order, $request, $data, $accessToken, $order_id);
}

/**
 * Retry submission normalized result
 */
function storeTipaxOrderAfterFailed($apiPath, $order, Request $request, stdClass $data, $accessToken, $order_id)
{
    $result = [
        'success'         => false,
        'messages'        => [],
        'order_id'        => $order_id,
        'tracking_codes'  => [],
        'tipax_order_id'  => null,
        'http_status'     => null,
    ];

    // Helper (no arrow functions)
    $extract_error_text = function ($resp) {
        if (is_wp_error($resp)) return $resp->get_error_message();
        if (is_object($resp)) {
            if (!empty($resp->body) && is_string($resp->body)) return $resp->body;
            if (!empty($resp->errors) && is_array($resp->errors)) {
                $msgs = [];
                foreach ($resp->errors as $e) {
                    if (!empty($e->errorMessage)) $msgs[] = $e->errorMessage;
                }
                return implode("\n", $msgs);
            }
            if (!empty($resp->message)) return $resp->message;
        }
        if (is_string($resp)) return $resp;
        return 'خطای نامشخص از تیپاکس.';
    };

    // already registered?
    $firstItem = null;
    foreach ($order->get_items() as $it) { $firstItem = $it; break; }
    $existing_tipax_code = $firstItem
        ? ($firstItem->get_meta("tipax_tracking_code") . $firstItem->get_meta("tipax_order_id"))
        : '';

    if ($existing_tipax_code) {
        $result['success']  = true;
        $result['messages'][] = 'سفارش قبلاً در تیپاکس ثبت شده است.';
        foreach ($order->get_items() as $item_id => $item) {
            $codes = trim((string)$item->get_meta("tipax_tracking_code"));
            if ($codes !== '') $result['tracking_codes'][] = $codes;
            if (!$result['tipax_order_id']) {
                $oid = $item->get_meta("tipax_order_id");
                if (!empty($oid)) $result['tipax_order_id'] = $oid;
            }
        }
        return $result;
    }

    // cookie reuse
    $tipaxOrderCookie = tipaxPluginCookie("tipaxOrder_" . $order_id);
    if ($tipaxOrderCookie) {
        $tipaxOrderInfo = json_decode(base64_decode($tipaxOrderCookie), false);

        if (!empty($tipaxOrderInfo->trackingCodes)) {
            $i = 0;
            foreach ($order->get_items() as $item_id => $item) {
                $quantity  = (int)$item->get_quantity();
                $codes_str = '';

                if ($quantity === 1) {
                    $code = strtoupper((string)($tipaxOrderInfo->trackingCodes[$i] ?? ''));
                    if ($code !== '') {
                        wc_add_order_item_meta($item_id, 'tipax_tracking_code', $code);
                        $result['tracking_codes'][] = $code;
                    }
                } else {
                    for ($j = 0; $j < $quantity; $j++) {
                        $code = strtoupper((string)($tipaxOrderInfo->trackingCodes[$j] ?? ''));
                        if ($code !== '') {
                            $codes_str .= $code . ' ';
                            $result['tracking_codes'][] = $code;
                        }
                    }
                    if ($codes_str !== '') {
                        wc_add_order_item_meta($item_id, 'tipax_tracking_code', trim($codes_str));
                    }
                }

                if (!empty($tipaxOrderInfo->orderId)) {
                    wc_add_order_item_meta($item_id, 'tipax_order_id', $tipaxOrderInfo->orderId);
                    $result['tipax_order_id'] = $tipaxOrderInfo->orderId;
                }

                $i++;
            }

            $result['success']    = true;
            $result['messages'][] = 'سفارش با موفقیت در تیپاکس ثبت شد (از کش/کوکی).';
            return $result;
        }
        // else: fall through and re-post
    }

    // API post
    $response = $request->apiRequest($apiPath, $data, 'POST', $accessToken);
    if (is_object($response) && isset($response->status) && (int)$response->status === 401) {
        $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);
        $response    = $request->apiRequest($apiPath, $data, 'POST', $accessToken);
    }

    if (is_object($response) && isset($response->status)) {
        $result['http_status'] = (int)$response->status;
    }
    if (is_wp_error($response)) {
        $result['messages'][] = $response->get_error_message();
        return $result;
    }

    if (is_object($response) && isset($response->errors) && is_array($response->errors)) {
        foreach ($response->errors as $error) {
            if (!empty($error->errorMessage)) $result['messages'][] = $error->errorMessage;
        }
        if (empty($result['messages'])) $result['messages'][] = 'خطای نامشخص از تیپاکس.';
        return $result;
    }

    if (is_object($response) && isset($response->status) && (int)$response->status === 400) {
        $result['messages'][] = $extract_error_text($response);
        return $result;
    }

    // success
    if (is_object($response) && !empty($response->trackingCodes) && is_array($response->trackingCodes)) {
        $expiration = current_time('timestamp') + (DAY_IN_SECONDS * 7);
        tipaxPluginCookie("tipaxOrder_" . $order_id, base64_encode(json_encode($response)), $expiration);

        $i = 0;
        foreach ($order->get_items() as $item_id => $item) {
            $quantity  = (int)$item->get_quantity();
            $codes_str = '';

            if ($quantity === 1) {
                $code = strtoupper((string)($response->trackingCodes[$i] ?? ''));
                if ($code !== '') {
                    wc_add_order_item_meta($item_id, 'tipax_tracking_code', $code);
                    $result['tracking_codes'][] = $code;
                }
            } else {
                for ($j = 0; $j < $quantity; $j++) {
                    $code = strtoupper((string)($response->trackingCodes[$j] ?? ''));
                    if ($code !== '') {
                        $codes_str .= $code . ' ';
                        $result['tracking_codes'][] = $code;
                    }
                }
                if ($codes_str !== '') {
                    wc_add_order_item_meta($item_id, 'tipax_tracking_code', trim($codes_str));
                }
            }

            if (!empty($response->orderId)) {
                wc_add_order_item_meta($item_id, 'tipax_order_id', $response->orderId);
                $result['tipax_order_id'] = $response->orderId;
            }

            $i++;
        }

        $result['success']    = true;
        $result['messages'][] = 'سفارش با موفقیت در تیپاکس ثبت شد.';
        return $result;
    }

    $result['messages'][] = $response;
    return $result;
}
