<?php

/**
 * Shpping Class
 * This class is for adding tipax shipping method to woocommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

if (class_exists('WC_Tipax_Shipping_Method')) {
    return;
} // Stop if the class already registered

class WC_Tipax_Shipping_Method extends WC_Shipping_Method
{
    /**
     * Constructor for your shipping class
     *
     * @access public
     * @return void
     */

    private $request;

    public function __construct($instance_id = 0)
    {
        $this->id = 'tipax_shipping_method'; // Id for your shipping method. Should be uunique.
        $this->instance_id = absint($instance_id);
        $this->domain = 'tipax_shipping_method';
        $this->title = __('Tipax Shipping Method', 'delivery-service-tipax-shipping'); // Title shown in FrontEnd
        $this->method_title = __('Tipax Shipping Method', 'delivery-service-tipax-shipping'); // Title shown in admin
        $this->method_description = __('Ship your orders with tipax', 'delivery-service-tipax-shipping'); // Description shown in admin

        $this->supports = array(
            'settings',
            'shipping-zones',
            'instance-settings',
            'instance-settings-modal'
        );

        $this->enabled = $this->get_option('enabled');

        $this->request = new Request();

        //check access token exists or no
        $accessToken = get_option('woocommerce_tipax_shipping_method_accesstoken', null);

        if (isset($accessToken) || $accessToken != null) {
            $this->accessToken = $accessToken;
        } else {
            $this->accessToken = null;
        }

        //check settings exists or no
        $settings = get_option('woocommerce_tipax_shipping_method_settings', null);

        if (isset($settings) || $settings != null) {
            $this->settings = $settings;
        } else {
            $this->settings = null;
        }

        $plugin_settings = get_option('woocommerce_tipax_shipping_method_settings', []);
        $collection_time = $plugin_settings['collection_time'] ?? 1;
        $this->collection_time = $collection_time;
        $this->address_method = $this->get_option('address_method');
        $this->lat_lon = $this->get_option('lat_lon');
        $this->checkout_method = $this->get_option('checkout_method');
        $this->shipping_address = get_option('tipax_shipping_address');
        $this->woocommerce_currency = get_woocommerce_currency();

        add_action('woocommerce_update_options_shipping_' . $this->id, array($this, 'process_admin_options'));
        $this->init();
    }

    /**
     * Is this method enabled?
     */
    public function is_enabled()
    {
        if ($this->get_option('enabled') == 'yes') {
            return true;
        } else {
            return false;
        }
    }
    /**
     * Is this method available?
     *
     * @param array $package Package.
     * @return bool
     */
    public function is_available($package): bool
    {
        $available = $this->is_enabled();

        // Country availability (legacy, for non-zone based methods).
        if (! $this->instance_id && $available) {
            $countries = is_array($this->countries) ? $this->countries : array();

            switch ($this->availability) {
                case 'specific':
                case 'including':
                    $available = in_array($package['destination']['country'], array_intersect($countries, array_keys(WC()->countries->get_shipping_countries())));
                    break;
                case 'excluding':
                    $available = in_array($package['destination']['country'], array_diff(array_keys(WC()->countries->get_shipping_countries()), $countries));
                    break;
                default:
                    $available = in_array($package['destination']['country'], array_keys(WC()->countries->get_shipping_countries()));
                    break;
            }
        }
        //package must have value!
        if (empty($package)) {
            $available = false;
            tipax_log("empty( package ): " . (int)$available);
        }

        //if country is not iran is not available
        if ($package['destination']['country'] != 'IR') {
            $available = false;
            tipax_log("package['destination']['country'] != 'IR': " . (int)$available);
        }

        //if accessToken not saved before is not available

        if (is_null($this->accessToken)) {
            $available = false;
            tipax_log("is_null(this->accessToken ): " . (int)$available);
        }

        $plugin_settings = get_option('woocommerce_tipax_shipping_method_settings', []);
        //if name is not confirmed is not available
        $senderOriginName = $plugin_settings['name'] ?? '';
        if (empty($senderOriginName)) {
            $available = false;
            tipax_log("is_null(senderOriginName ): " . (int)$available);
        }

        //if phone is not confirmed is not available
        $senderOriginPhone = $plugin_settings['phone'] ?? '';
        if (empty($senderOriginPhone)) {
            $available = false;
            tipax_log("is_null(senderOriginPhone ): " . (int)$available);
        }

        //if lat_lon is not selected is not available
        // $originLatLon = $this->get_option("lat_lon");
        // $lat = null;
        // $lon = null;
        // if($originLatLon != null){
        //     $lat_lon = explode(",",$originLatLon);
        //     $lat = $lat_lon[0];
        //     $lon = $lat_lon[1];
        // }

        // //the lat & lon is required
        // if(is_null( $originLatLon ) || $lat == null || $lon == null ){
        //     $available = false;
        //     tipax_log("is_null( originLatLon ): ".(int)$available);
        // }

        //check origin postalcode and validate it
        // $originPostalCode = get_option( 'woocommerce_store_postcode' );
        // $validatePostalCode = $originPostalCode ? postalcodeValidate($originPostalCode) : true;
        // if ($validatePostalCode == null ) {
        //     $available = false;
        //     tipax_log("is_null( validatePostalCode ): ". (int)$available . " originPostalCode: ".$originPostalCode);
        // }

        //validate and check destination postalcode
        // $destinationPostalCode = $package["destination"]["postcode"];
        // $validateDestinationPostalCode = $destinationPostalCode ? postalcodeValidate($destinationPostalCode) : true;
        // if ($validateDestinationPostalCode == null) {
        //     $available = false;
        //     tipax_log("is_null( validateDestinationPostalCode ): " . (int)$available . " DestinationPostalCode: " . $destinationPostalCode);
        // }

        if ($available) {
            tipax_log("No problem about available variable: " . (int)$available);
        }

        return apply_filters('woocommerce_shipping_' . $this->id . '_is_available', $available, $package, $this);
    }
    /**
     * Init your settings
     *
     * @access public
     * @return void
     */
    public function init()
    {
        // Load the settings API
        $this->init_instance_form_fields(); // This is part of the instant settings API. showing in modal
        $this->init_form_fields(); // This is part of the settings API. Override the method to add your own settings
        $this->init_settings(); // This is part of the settings API. Loads settings you previously init.

        // Save settings in admin if you have any defined
        add_action('woocommerce_update_options_shipping_' . $this->id, array($this, 'process_admin_options'));
        //add assets needed
        add_action('admin_footer', array('WC_Tipax_Shipping_Method', 'enqueue_admin_js'), 10); // Priority needs to be higher than wc_print_js (25).
    }

    public function init_form_fields()
    {
        $this->checkApiKey();
        $this->form_fields = array(
            'enabled' => array(
                'title'   => __('Enable/Disable', 'delivery-service-tipax-shipping'),
                'type'    => 'checkbox',
                'label'   => __('Enable this shipping method', 'delivery-service-tipax-shipping'),
                'default' => 'yes',
            ),
            'cities' => array(
                'title'   => __('Only Tipax cities?', 'delivery-service-tipax-shipping'),
                'type'    => 'checkbox',
                'label'   => __('Do you want Tipax fills your cities?', 'delivery-service-tipax-shipping'),
                'default' => 'no',
            ),
            'username' => array(
                'title' => __('Username', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'required' => true,
                'placeholder' => __('Enter Your Tipax Username', 'delivery-service-tipax-shipping'),
                'description' => __('Username of your tipax account', 'delivery-service-tipax-shipping'),
                'desc_tip' => true
            ),
            'password' => array(
                'title' => __('Password', 'delivery-service-tipax-shipping'),
                'type' => 'password',
                'required' => true,
                'placeholder' => __('Enter Your Tipax Password', 'delivery-service-tipax-shipping'),
                'description' => __('password of your tipax account', 'delivery-service-tipax-shipping'),
                'desc_tip' => true
            ),
            'apiKey' => array(
                'title' => __('API KEY', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'required' => true,
                'placeholder' => __('Enter Your Tipax Api Key which is located in profile page', 'delivery-service-tipax-shipping'),
                'description' => __('this Tipax Api Key is located in profile page', 'delivery-service-tipax-shipping'),
                'desc_tip' => true
            ),
        );
    }

    public function init_instance_form_fields()
    {
        $addresses = [];
        $custom_attributes_addresses = [];
        $custom_attributes_addresses_data = [];
        $addresses_objects = $this->loadAddresses();
        $cities_objects = $this->loadCities();
        $all_addresses = array();
        if ($addresses_objects != null) {
            $all_addresses[] = $addresses_objects->address;
            foreach ($all_addresses as $addresses_object) {
                $addresses[$addresses_object->id] = $addresses_object->fullAddress;

                $custom_attributes_addresses[]["id"] = $addresses_object->id;
                $custom_attributes_addresses[]["cityId"] = $addresses_object->cityId;
                $custom_attributes_addresses[]["stateId"] = $addresses_object->provinceId;
                $custom_attributes_addresses[]["fullAddress"] = $addresses_object->fullAddress;
            }

            $custom_attributes_addresses_json = json_encode($custom_attributes_addresses);
            $custom_attributes_addresses_data["data"] = $custom_attributes_addresses_json;
        } else {
            $custom_attributes_addresses_data["data"] = "";
            $addresses = [];
        }

        $this->instance_form_fields = array(
            'lat_lon' => array(
                'title' => __('pick address on map (Required)', 'delivery-service-tipax-shipping'),
                'description' => __('click on map to select the point', 'delivery-service-tipax-shipping'),
                'type' => 'map',
                'class' => 'leaflet-map-main',
                'id' => 'leaflet',
                'default' => null,
                'label' => __('shipping address', 'delivery-service-tipax-shipping'),
            ),
            'title' => array(
                'title' => __('Address Title', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'class' => 'title',
                'id' => 'title',
                'default' => null,
                'label' => __('Title', 'delivery-service-tipax-shipping'),
            ),
            'postal_code' => array(
                'title' => __('Postal Code', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'class' => 'postal_code',
                'id' => 'postal_code',
                'default' => null,
                'label' => __('Correct Postal Code', 'delivery-service-tipax-shipping'),
            ),
            'floor' => array(
                'title' => __('Floor', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'class' => 'floor',
                'id' => 'floor',
                'default' => null,
                'label' => __('Floor', 'delivery-service-tipax-shipping'),
            ),
            'unit' => array(
                'title' => __('Unit', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'class' => 'unit',
                'id' => 'unit',
                'default' => null,
                'label' => __('Unit', 'delivery-service-tipax-shipping'),
            ),
            'city' => array(
                'title' => __('city', 'delivery-service-tipax-shipping'),
                'description' => __('City of Origin', 'delivery-service-tipax-shipping'),
                'type' => 'select',
                'default' => null,
                'class' => 'js-example-basic-single',
                'label' => __('City', 'delivery-service-tipax-shipping'),
                'options' => $cities_objects
            ),
            'address' => array(
                'title' => __('Address', 'delivery-service-tipax-shipping'),
                'type' => 'text',
                'class' => 'address',
                'id' => 'address',
                'default' => null,
                'label' => __('Address', 'delivery-service-tipax-shipping'),
            ),
        );
    }

    public function generate_map_html($key, $data)
    {

        $field_key = $this->get_field_key($key);
        $defaults  = array(
            'title'             => '',
            'disabled'          => false,
            'class'             => '',
            'css'               => '',
            'placeholder'       => '',
            'type'              => 'text',
            'desc_tip'          => false,
            'description'       => '',
            'custom_attributes' => array(),
        );

        $data = wp_parse_args($data, $defaults);

        ob_start();
?>
        <tr valign="top">
            <th scope="row" class="titledesc">
                <label for="<?php echo esc_attr($field_key); ?>"><?php echo wp_kses_post($data['title']); ?> <?php echo $this->get_tooltip_html($data); // WPCS: XSS ok. 
                                                                                                                ?></label>
            </th>
            <td class="forminp">
                <fieldset>
                    <legend class="screen-reader-text"><span><?php echo wp_kses_post($data['title']); ?></span></legend>

                    <div class="input-text regular-input <?php echo esc_attr($data['class']); ?>" type="<?php echo esc_attr($data['type']); ?>" name="<?php echo esc_attr($field_key); ?>" id="woocommerce_tipax_shipping_method_map" style="<?php echo esc_attr($data['css']); ?>" placeholder="<?php echo esc_attr($data['placeholder']); ?>" <?php disabled($data['disabled'], true); ?> <?php echo $this->get_custom_attribute_html($data); // WPCS: XSS ok. 
                                                                                                                                                                                                                                                                                                                                                                                            ?>></div>
                    <input type="hidden" class="input-text regular-input" name="<?php echo esc_attr($field_key); ?>" id="map_lat_lon" value="<?php echo esc_attr($this->get_option($key)); ?>">
                    <?php echo $this->get_description_html($data); // WPCS: XSS ok. 
                    ?>
                </fieldset>
            </td>
        </tr>

        <script>
            var map = L.map('woocommerce_tipax_shipping_method_map').setView([33.6371298, 53.2590134], 5);

            var lat_lon_selected = jQuery("#map_lat_lon").val();

            var lat_lon_is_valid = false;
            if (lat_lon_selected.indexOf(",") >= 0) {
                lat_lon_is_valid = true;
            }

            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: 'Tipax shipping plugin',
                center: [33.6371298, 53.2590134],
                zoom: 10
            }).addTo(map);

            var marker;

            if (lat_lon_selected && lat_lon_is_valid === true) {
                if (marker) {
                    map.removeLayer(marker);
                }

                var iconOptions = {
                    iconUrl: '<?php echo TSM_ASSETS_URL ?>images/tipax-pointer.png',
                    iconSize: [50, 50]
                }

                // Creating a custom icon
                var customIcon = L.icon(iconOptions);
                var markerOptions = {
                    title: "MyLocation",
                    clickable: false,
                    draggable: false,
                    icon: customIcon
                }
                // Creating a marker
                if (lat_lon_selected) {
                    lat_lon_selected = lat_lon_selected.split(',');
                    marker = L.marker(lat_lon_selected, markerOptions).addTo(map);
                }
            }

            map.on('click', function(e) {
                jQuery("#map_lat_lon").val("");
                console.log("Lat, Lon : " + e.latlng.lat + ", " + e.latlng.lng);

                if (marker) {
                    map.removeLayer(marker);
                }

                var iconOptions = {
                    iconUrl: '<?php echo TSM_ASSETS_URL ?>images/tipax-pointer.png',
                    iconSize: [50, 50]
                }

                // Creating a custom icon
                var customIcon = L.icon(iconOptions);
                var markerOptions = {
                    title: "MyLocation",
                    clickable: false,
                    draggable: false,
                    icon: customIcon
                }
                // Creating a marker
                marker = L.marker([e.latlng.lat, e.latlng.lng], markerOptions);

                jQuery("#map_lat_lon").val(e.latlng.lat + "," + e.latlng.lng);
                // Adding marker to the map
                marker.addTo(map);
            });
            jQuery(document).ready(function() {
                jQuery('.js-example-basic-single').select2();
            });
        </script>
    <?php
        return ob_get_clean();
    }

    public function generate_radio_html($key, $data)
    {
        $field_key = $this->get_field_key($key);
        $defaults  = array(
            'title'             => '',
            'disabled'          => false,
            'class'             => '',
            'css'               => '',
            'placeholder'       => '',
            'type'              => 'text',
            'desc_tip'          => false,
            'description'       => '',
            'custom_attributes' => array(),
        );

        $data = wp_parse_args($data, $defaults);

        ob_start();
    ?>
        <tr valign="top">
            <th scope="row" class="titledesc">
                <label for="<?php echo esc_attr($field_key); ?>"><?php echo wp_kses_post($data['title']); ?> <?php echo $this->get_tooltip_html($data); // WPCS: XSS ok. 
                                                                                                                ?></label>
            </th>
            <td class="forminp">
                <fieldset>
                    <legend class="screen-reader-text"><span><?php echo wp_kses_post($data['title']); ?></span></legend>

                    <!--<input style="width: 16px !important; min-width: unset !important;" type="radio" id="radio_example_one" name="woocommerce_tipax_shipping_method_address_method" value="1" <?php if (esc_attr($this->get_option($key)) == 1) {
                                                                                                                                                                                                        echo "checked='true'";
                                                                                                                                                                                                    } ?>>-->
                    <!--<label for="radio_example_one"><?php echo __('From woocommerce shop information', 'delivery-service-tipax-shipping') ?></label>-->

                    <input style="width: 16px !important; min-width: unset !important;" type="radio" id="radio_example_two" name="woocommerce_tipax_shipping_method_address_method" value="2" <?php if (esc_attr($this->get_option($key)) == 2) {
                                                                                                                                                                                                    echo "checked='true'";
                                                                                                                                                                                                } ?>>
                    <label for="radio_example_two"><?php echo __('Pick up from map', 'delivery-service-tipax-shipping') ?></label>

                    <input style="width: 16px !important; min-width: unset !important;" type="radio" id="radio_example_three" name="woocommerce_tipax_shipping_method_address_method" value="3" <?php if (esc_attr($this->get_option($key)) == 3) {
                                                                                                                                                                                                    echo "checked='true'";
                                                                                                                                                                                                } ?>>
                    <label for="radio_example_three"><?php echo __('From defined addresses', 'delivery-service-tipax-shipping') ?></label>

                    <?php echo $this->get_description_html($data); // WPCS: XSS ok.
                    ?>
                </fieldset>
            </td>
        </tr>
<?php
        return ob_get_clean();
    }

    public static function enqueue_admin_js()
    {

        if (is_checkout()) {
            return false;
        }

        wp_register_script('leaflet_map', TSM_ASSETS_URL . 'js/leaflet.js', ['jquery'], '4.0.3');
        wp_enqueue_script('leaflet_map', "", "", "", true);
        wp_register_style('leaflet_map_css', TSM_ASSETS_URL . 'css/leaflet.css');
        wp_enqueue_style('leaflet_map_css');
        wp_register_style('select2css', 'css/select2.css', false, '1.0', 'all');
        wp_register_script('select2', 'js/select2.js', array('jquery'), '1.0', true);
        wp_enqueue_style('select2css');
        wp_enqueue_script('select2');
    }

    public function add_rate($args = array())
    {
        $args = apply_filters(
            'woocommerce_shipping_method_add_rate_args',
            wp_parse_args(
                $args,
                array(
                    'id'             => $this->get_rate_id(), // ID for the rate. If not passed, this id:instance default will be used.
                    'label'          => '', // Label for the rate.
                    'cost'           => '0', // Amount or array of costs (per item shipping).
                    'taxes'          => '', // Pass taxes, or leave empty to have it calculated for you, or 'false' to disable calculations.
                    'calc_tax'       => 'per_order', // Calc tax per_order or per_item. Per item needs an array of costs.
                    'meta_data'      => array(), // Array of misc meta data to store along with this rate - key value pairs.
                    'package'        => false, // Package array this rate was generated for @since 2.6.0.
                    'price_decimals' => wc_get_price_decimals(),
                )
            ),
            $this
        );

        // ID and label are required.
        if (!$args['id'] || !$args['label']) {
            return;
        }

        // Total up the cost.
        $total_cost = is_array($args['cost']) ? array_sum($args['cost']) : $args['cost'];
        $taxes      = $args['taxes'];

        // Taxes - if not an array and not set to false, calc tax based on cost and passed calc_tax variable. This saves shipping methods having to do complex tax calculations.
        if (!is_array($taxes) && false !== $taxes && $total_cost > 0 && $this->is_taxable()) {
            $taxes = 'per_item' === $args['calc_tax'] ? $this->get_taxes_per_item($args['cost']) : WC_Tax::calc_shipping_tax($total_cost, WC_Tax::get_shipping_tax_rates());
        }

        // Round the total cost after taxes have been calculated.
        $total_cost = wc_format_decimal($total_cost, $args['price_decimals']);

        // Create rate object.
        $rate = new WC_Shipping_Rate();
        $rate->set_id($args['id']);
        $rate->set_method_id($this->id);
        $rate->set_instance_id($this->instance_id);
        $rate->set_label($args['label']);
        $rate->set_cost($total_cost);
        $rate->set_taxes($taxes);

        if (!empty($args['meta_data'])) {
            foreach ($args['meta_data'] as $key => $value) {
                $rate->add_meta_data($key, $value);
            }
        }

        // Store package data.
        if ($args['package']) {
            $items_in_package = array();
            foreach ($args['package']['contents'] as $item) {
                $product            = $item['data'];
                $items_in_package[] = $product->get_name() . ' &times; ' . $item['quantity'];
            }
            $rate->add_meta_data(__('Items', 'woocommerce'), implode(', ', $items_in_package));
        }
        $this->rates[$args['id']] = apply_filters('woocommerce_shipping_method_add_rate', $rate, $args, $this);
    }
public function calculate_shipping($package = array())
{
    $cost = 0;

    // --- Store location & destination ---
    $storeRawCountry = get_option('woocommerce_default_country'); // e.g. IR:THR
    $splitSplit = explode(':', $storeRawCountry);
    $storeCountry = $splitSplit[0] ?? '';
    $storeState   = $splitSplit[1] ?? '';
    $storeCity    = get_option('woocommerce_store_city');

    $packageCity    = $package['destination']['city']    ?? '';
    $packageCountry = $package['destination']['country'] ?? '';
    $packageState   = $package['destination']['state']   ?? '';

    // Only IR supported
    if (strtoupper($storeCountry) !== 'IR') {
        return false;
    }

    $originStateReal   = WC()->countries->get_states($storeCountry)[$storeState] ?? null;
    $packageStateReal  = WC()->countries->get_states($packageCountry)[$packageState] ?? null;

    // Resolve Tipax destination city code
    $destinationCityId = get_city_code($packageCity, $packageStateReal);
    if ($destinationCityId === null) {
        tipax_log("packageCity: {$packageCity} storeCity: {$storeCity} destinationCityId: " . var_export($destinationCityId, true));
        return false;
    }

    // ---- Address method (address-based only, as requested) ----
    $addressMethod   = 3;
    $originAddressId = get_option('tipax_shipping_address');
    if ($addressMethod == 3 && empty($originAddressId)) {
        tipax_log('originAddressId: ' . (int)$originAddressId);
        return false;
    }

    // ---- Settings / options ----
    $plugin_settings          = get_option('woocommerce_tipax_shipping_method_settings', []);
    $this->checkout_method    = $plugin_settings['checkout_method'] ?? null;
    if ($this->checkout_method === null) {
        $this->checkout_method = 10;
    }

    // Allowed pack types (kept for packType selection below)
    $allowed_packtypes = ['10', '20', '50']; // 10: پاکت, 20: بسته, 50: مینی پک
    $packType = in_array(($plugin_settings['packtype'] ?? '20'), $allowed_packtypes, true)
        ? (int)$plugin_settings['packtype']
        : 20;

    // Optional avg fallbacks from settings
    $avgWeight = (isset($plugin_settings['avg_weight']) && is_numeric($plugin_settings['avg_weight'])) ? (float)$plugin_settings['avg_weight'] : null; // kg
    $avgLength = (isset($plugin_settings['avg_length']) && is_numeric($plugin_settings['avg_length'])) ? (float)$plugin_settings['avg_length'] : null; // cm
    $avgWidth  = (isset($plugin_settings['avg_width'])  && is_numeric($plugin_settings['avg_width']))  ? (float)$plugin_settings['avg_width']  : null; // cm
    $avgHeight = (isset($plugin_settings['avg_height']) && is_numeric($plugin_settings['avg_height'])) ? (float)$plugin_settings['avg_height'] : null; // cm

    $weight_unit     = get_option('woocommerce_weight_unit');
    $dimension_unit  = get_option('woocommerce_dimension_unit');

    // --- Converters ---
    $toKg = function ($val) use ($weight_unit) {
        $v = (float)$val;
        if ($v <= 0) return 0.0;
        switch (strtolower($weight_unit)) {
            case 'kg': return $v;
            case 'g':  return $v / 1000.0;
            case 'lbs':
            case 'lb': return $v * 0.45359237;
            case 'oz': return $v * 0.028349523125;
            default:   return $v; // assume kg
        }
    };

    // Convert Woo dimension -> cm  (FIX: correct ratios)
    switch (strtolower($dimension_unit)) {
        case 'm':  $unit_ratio = 100.0;   break; // m -> cm
        case 'cm': $unit_ratio = 1.0;     break; // cm -> cm
        case 'mm': $unit_ratio = 0.1;     break; // mm -> cm
        case 'in': $unit_ratio = 2.54;    break; // inch -> cm
        case 'yd': $unit_ratio = 91.44;   break; // yard -> cm
        default:   $unit_ratio = 1.0;     break;
    }

    // Helper: priority picker (product → settings → hard default)
    $pick = function ($productVal, $settingsVal, $absoluteDefault) {
        if ($productVal !== null && $productVal > 0) return (float)$productVal;
        if ($settingsVal !== null && $settingsVal > 0) return (float)$settingsVal;
        return (float)$absoluteDefault;
    };

    // ---- Gather cart totals & items ----
    $cart_subtotal_store = WC()->cart ? (float)WC()->cart->get_subtotal() : 0.0;

    $total_cart_weight_kg  = 0.0;
    $total_cart_volume_cm3 = 0.0;
    $total_item_qty        = 0;
    $single_item_snapshot  = null; // store single item dims/weight if qty==1 only

    if (!empty($package['contents'])) {
        foreach ($package['contents'] as $ci) {
            $qty = (int)($ci['quantity'] ?? 0);
            if ($qty <= 0) continue;

            $p = !empty($ci['variation_id'])
                ? wc_get_product($ci['variation_id'])
                : wc_get_product($ci['product_id'] ?? 0);
            if (!$p) continue;

            $rawW = is_numeric($p->get_weight()) ? (float)$p->get_weight() : null;
            $rawL = is_numeric($p->get_length()) ? (float)$p->get_length() : null;
            $rawH = is_numeric($p->get_height()) ? (float)$p->get_height() : null;
            $rawD = is_numeric($p->get_width())  ? (float)$p->get_width()  : null;

            // Apply priority (product → avg → default)
            $item_weight_logic = $pick($rawW, $avgWeight, 6.0);  // kg logical
            $item_length_logic = $pick($rawL, $avgLength, 45.0); // cm logical
            $item_height_logic = $pick($rawH, $avgHeight, 45.0); // cm logical
            $item_width_logic  = $pick($rawD, $avgWidth,  45.0); // cm logical

            // Convert to kg/cm for aggregation
            $item_weight_kg = $toKg($item_weight_logic);
            $L_cm = $item_length_logic * $unit_ratio;
            $H_cm = $item_height_logic * $unit_ratio;
            $W_cm = $item_width_logic  * $unit_ratio;

            // Basic sanity check: block absurd values
            if ($L_cm <= 0 || $W_cm <= 0 || $H_cm <= 0 || $item_weight_kg <= 0) {
                tipax_log("Tipax shipping: some dimensions/weight are zero or invalid.");
                return false;
            }

            $total_cart_weight_kg  += $item_weight_kg * $qty;
            $total_cart_volume_cm3 += ($L_cm * $W_cm * $H_cm) * $qty;
            $total_item_qty        += $qty;

            if ($total_item_qty === $qty && $qty === 1) {
                // This is the only item in the cart (so far). Keep snapshot for single-item precision.
                $single_item_snapshot = [
                    'weight_kg' => $item_weight_kg,
                    'L_cm'      => $L_cm,
                    'W_cm'      => $W_cm,
                    'H_cm'      => $H_cm,
                    // price (for packageValue)
                    'price'     => (float)$p->get_price(),
                ];
            }
        }
    }

    // --- Light shipment threshold (for packType/packing choice) ---
    $threshold_store = 4000000.0; // in Toman by your original logic
    if (strtoupper($this->woocommerce_currency) === 'IRR') {
        $threshold_store *= 10; // IRR
    }
    $is_light_shipment = (
        $total_cart_weight_kg > 0 &&
        $total_cart_weight_kg < 2.0 &&
        $cart_subtotal_store < $threshold_store
    );
    tipax_log('total_cart_weight_kg=' . $total_cart_weight_kg . ' | is_light_shipment=' . (int)$is_light_shipment);
    tipax_log("weightUnit: {$weight_unit} | dimensionUnit: {$dimension_unit}");

    // ---- Load & validate origin address once (addressMethod = 3) ----
    $originCityETipaxId = null;
    if ($addressMethod == 3) {
        if (!$this->accessToken) return false;

        // Try cached transient first
        $addresses_objects = $this->loadAddresses();
        $dataAddr = [];
        $responseAddr = $this->request->apiRequest('Addresses', $dataAddr, "GET", $this->accessToken);
        if (!isset($responseAddr[0])) {
            return false;
        }

        $allAddresses = [];
        foreach ($responseAddr as $addr) {
            $allAddresses[$addr->id] = [
                'originId'    => $addr->addressesId,
                'fullAddress' => $addr
            ];
        }

        // Ensure the selected originAddressId exists; otherwise pick the first and save.
        $selected = null;
        foreach ($allAddresses as $row) {
            if ((string)$row['originId'] === (string)$originAddressId) {
                $selected = $row['fullAddress'];
                break;
            }
        }
        if (!$selected) {
            // fallback to addresses_objects->id if present, else the first one
            $fallback = isset($addresses_objects->id, $allAddresses[$addresses_objects->id]) ? $allAddresses[$addresses_objects->id] : reset($allAddresses);
            $originAddressId = $fallback['originId'];
            if (get_option('tipax_shipping_address') !== false) {
                update_option('tipax_shipping_address', $originAddressId);
            } else {
                add_option('tipax_shipping_address', $originAddressId, '', 'yes');
            }
            delete_transient('tipax_addresses');
            set_transient('tipax_addresses', $fallback['fullAddress'], 12 * HOUR_IN_SECONDS);
            $selected = $fallback['fullAddress'];
            tipax_log('Your new AddressId is :' . get_option('tipax_shipping_address'));
        }

        // For serviceId decision
        if (isset($selected->address->cityETipaxId)) {
            $originCityETipaxId = (string)$selected->address->cityETipaxId;
        }
    }

    // ---- Build packageInputs (single precise vs aggregated minimal) ----
    $data = new stdClass();
    $data->packageInputs = [];

    // Max constraints
    $MAX_DIM_CM      = 200.0;
    $PER_ITEM_MAX_KG = 100.0;
    $CUBE_CM3        = $MAX_DIM_CM * $MAX_DIM_CM * $MAX_DIM_CM;

    // Helper to set pack meta (packingId, packageContentId, packType) based on a package’s dims & weight
    $apply_pack_meta = function ($pkg) {
        // default: box/goods
        $pkg->packingId         = 504;
        $pkg->packageContentId  = 9;
        $pkg->packType          = 20;

        // Light letter/minipack condition (<= 10 cm sides and < 2 kg) — keep your earlier logic
        if (
            $pkg->weight > 0 && $pkg->weight < 2.0 &&
            $pkg->length <= 10 && $pkg->width <= 10 && $pkg->height <= 5
        ) {
            $pkg->packingId        = 512;
            $pkg->packageContentId = 10;
            $pkg->packType         = 50;
        }
        return $pkg;
    };

    // Helper to create a packageInput object for API
    $make_pkg = function ($weight_kg, $side_cm, $package_value_store) use ($addressMethod, $originAddressId, $destinationCityId, $apply_pack_meta) {
        $pkg = new stdClass();
        if ($addressMethod == 3) {
            $pkg->origin = new stdClass();
            $pkg->origin->Id = $originAddressId;
        }
        $pkg->destination = new stdClass();
        $pkg->destination->cityId = $destinationCityId;

        // Units for API
        $pkg->weight = (float)$weight_kg;
        $pkg->length = (float)$side_cm;
        $pkg->width  = (float)$side_cm;
        $pkg->height = (float)$side_cm;

        // packageValue: respect currency (IRT ×10 -> IRR)
        $pkg->packageValue = (strtoupper($this->woocommerce_currency) === 'IRT')
            ? (float)($package_value_store * 10.0)
            : (float)$package_value_store;

        $pkg->distributionType = 10;
        $pkg->pickupType       = 10;
        $pkg->paymentType      = $this->checkout_method;

        $pkg = $apply_pack_meta($pkg);
        return $pkg;
    };

    // Branch A: single item (qty = 1) → use its exact dims/weight; still split if exceeds limits
    if ($total_item_qty === 1 && $single_item_snapshot) {
        $w_kg = $single_item_snapshot['weight_kg'];
        $L    = $single_item_snapshot['L_cm'];
        $W    = $single_item_snapshot['W_cm'];
        $H    = $single_item_snapshot['H_cm'];

        // A conservative cube side approximation for API (use max dimension)
        $max_side = max($L, $W, $H);
        $vol_cm3  = $L * $W * $H;

        // Determine number of packages needed by constraints
        $n_by_kg = (int)ceil($w_kg / $PER_ITEM_MAX_KG);
        $n_by_dm = (int)ceil(($max_side > 0 ? ($max_side / $MAX_DIM_CM) : 0.0));
        $n_by_vol= (int)ceil($vol_cm3 / $CUBE_CM3);
        $num_pkgs = max(1, max($n_by_kg, max($n_by_dm, $n_by_vol)));

        $pi_weight = $w_kg / $num_pkgs;
        $pi_value  = $single_item_snapshot['price'] / $num_pkgs;

        $remaining_vol = $vol_cm3;

        for ($i = 1; $i <= $num_pkgs; $i++) {
            if ($i < $num_pkgs) {
                $side = $MAX_DIM_CM;
                $pkg_weight = $pi_weight;
                $pkg_value  = $pi_value;
                $remaining_vol -= $CUBE_CM3;
                if ($remaining_vol < 0) $remaining_vol = 0;
            } else {
                // Last package: volume-based side
                $side = ($remaining_vol > 0) ? pow($remaining_vol, 1/3) : $MAX_DIM_CM;
                if ($side > $MAX_DIM_CM) $side = $MAX_DIM_CM;
                $pkg_weight = $pi_weight;
                $pkg_value  = $pi_value;
            }
            $pkg = $make_pkg($pkg_weight, ceil($side), $pkg_value);
            $data->packageInputs[] = $pkg;
        }
    }
    // Branch B: multiple items or qty > 1 → aggregate to minimal packages
    else {
        if ($total_cart_weight_kg <= 0 || $total_cart_volume_cm3 <= 0) {
            tipax_log("Tipax shipping: invalid cart totals.");
            return false;
        }

        // Determine minimal package count based on constraints
        $number_by_kg  = (int)ceil($total_cart_weight_kg / $PER_ITEM_MAX_KG);
        $number_by_vol = (int)ceil($total_cart_volume_cm3 / $CUBE_CM3);
        $package_count = max(1, max($number_by_kg, $number_by_vol));

        // Also, don't exceed total item count (optional, but safe)
        if ($total_item_qty > 0) {
            $package_count = min($package_count, $total_item_qty);
        }

        $pi_weight = $total_cart_weight_kg  / $package_count;
        $pi_value  = $cart_subtotal_store   / $package_count;

        $remaining_vol = $total_cart_volume_cm3;

        for ($i = 1; $i <= $package_count; $i++) {
            if ($i < $package_count) {
                $side = $MAX_DIM_CM; // full cube
                $pkg_weight = $pi_weight;
                $pkg_value  = $pi_value;
                $remaining_vol -= $CUBE_CM3;
                if ($remaining_vol < 0) $remaining_vol = 0;
            } else {
                // last package gets the remaining volume
                $side = ($remaining_vol > 0) ? pow($remaining_vol, 1/3) : $MAX_DIM_CM;
                if ($side > $MAX_DIM_CM) $side = $MAX_DIM_CM;
                $pkg_weight = $pi_weight;
                $pkg_value  = $pi_value;
            }

            // Guard against limits
            if ($pkg_weight > $PER_ITEM_MAX_KG) {
                // extra split if weight rounding created an overflow
                $overflow_parts = (int)ceil($pkg_weight / $PER_ITEM_MAX_KG);
                $split_weight   = $pkg_weight / $overflow_parts;
                for ($k = 1; $k <= $overflow_parts; $k++) {
                    $pkg = $make_pkg($split_weight, ceil($side), $pkg_value / $overflow_parts);
                    $data->packageInputs[] = $pkg;
                }
            } else {
                $pkg = $make_pkg($pkg_weight, ceil($side), $pkg_value);
                $data->packageInputs[] = $pkg;
            }
        }
    }

    // ---- ServiceId (same logic as before): 7 for intra-city, 5 otherwise ----
    // If we know origin city eTipax id and it equals destination, mark serviceId=7 else 5
    $serviceIdForAll = 5;
    if ($originCityETipaxId !== null && (string)$originCityETipaxId === (string)$destinationCityId) {
        $serviceIdForAll = 7;
    }
    foreach ($data->packageInputs as $pkg) {
        $pkg->serviceId = $serviceIdForAll;
    }

    // ---- Call Pricing API once (address method = 3 uses the same endpoint in your code) ----
    $response = $this->request->apiRequest('Pricing/WithOriginAddressId', $data, 'POST', $this->accessToken);
    if (!$response || isset($response->errors)) {
        return false;
    }

    // NOTE: Tipax returns a single aggregated price object even for multiple packageInputs
    // Handle all four rate types, exactly as you had — but without duplication.
    $rates_map = [
        'regularPlusRate' => null,
        'expressRate'     => null,
        'airExpressRate'  => null,
        'regularRate'     => null,
    ];
    // ✅ Handle both single-object and multi-object array response
$apiResults = is_array($response) ? $response : [$response];

// Define rate properties we support
$rates_map = ['regularPlusRate', 'expressRate', 'airExpressRate', 'regularRate'];

// We’ll aggregate prices for identical serviceIds across packs
$aggregatedRates = [];

foreach ($apiResults as $item) {
    if (!is_object($item)) continue;

    foreach ($rates_map as $prop) {
        if (!isset($item->{$prop}) || !is_object($item->{$prop})) continue;
        $rateObj = $item->{$prop};
        if (!isset($rateObj->finalPrice)) continue;

        $sid = $rateObj->serviceId ?? 0;
        if (!isset($aggregatedRates[$sid])) {
            $aggregatedRates[$sid] = [
                'serviceTitle' => $rateObj->serviceTitle ?? '',
                'finalPrice'   => 0,
                'serviceId'    => $sid,
            ];
        }

        // sum the total across packs
        $aggregatedRates[$sid]['finalPrice'] += (float)$rateObj->finalPrice;
    }
}

// --- Now process each aggregated rate exactly like before ---
foreach ($aggregatedRates as $sid => $rateData) {
    $cost = round((float)$rateData['finalPrice']);
    $currentPersianCurrency = 'ریال';

    if (strtoupper($this->woocommerce_currency) === 'IRT') {
        $cost = ceil($cost / 5) * 5;
        $cost /= 10;
        $cost = (int)$cost;
        $currentPersianCurrency = 'تومان';
    }

    $shippingServiceTitle = $rateData['serviceTitle'];

    // checkout_method handling (20: COD/postpay, 50: wallet, 100: custom flat)
    if (in_array($this->checkout_method, [20, 50, 100], true)) {
        $currentCost = number_format($cost) . ' ' . $currentPersianCurrency;

        if ($this->checkout_method == 20) {
            $serviceDesc = "(پس کرایه ، مبلغ {$currentCost})";
            $cost = 0;
        } elseif ($this->checkout_method == 50) {
            $serviceDesc = "(پرداخت از کیف پول مدیر فروشگاه، مبلغ {$currentCost})";
            $cost = 0;
        } elseif ($this->checkout_method == 100) {
            $inner_flat = (int)($plugin_settings['inner_flat_price'] ?? 0);
            $outer_flat = (int)($plugin_settings['outer_flat_price'] ?? 0);
            $inner_min  = (int)($plugin_settings['inner_min_cart_free_shipping'] ?? 0);
            $outer_min  = (int)($plugin_settings['outer_min_cart_free_shipping'] ?? 0);
            $cart_total = WC()->cart ? (float)WC()->cart->get_subtotal() : 0;

            if (strpos($shippingServiceTitle, 'درون شهری') !== false) {
                if ($cart_total >= $inner_min && $inner_min > 0) {
                    $cost = 0;
                    $serviceDesc = "(ارسال رایگان درون‌شهری)";
                } else {
                    $cost = $inner_flat;
                    $serviceDesc = "(درون‌شهری، مبلغ ثابت {$inner_flat} تومان)";
                }
            } elseif (strpos($shippingServiceTitle, 'بین شهری') !== false) {
                if ($cart_total >= $outer_min && $outer_min > 0) {
                    $cost = 0;
                    $serviceDesc = "(ارسال رایگان بین‌شهری)";
                } else {
                    $cost = $outer_flat;
                    $serviceDesc = "(بین‌شهری، مبلغ ثابت {$outer_flat} تومان)";
                }
            } else {
                $serviceDesc = "(مبلغ نامشخص)";
                $cost = 0;
            }
        }

        $shippingServiceTitle = "تیپاکس: {$serviceDesc} {$shippingServiceTitle}";
    } else {
        $shippingServiceTitle = "تیپاکس {$shippingServiceTitle}";
    }

    $rate = array(
        'id'        => $this->id . $rateData['serviceId'],
        'label'     => $shippingServiceTitle,
        'meta_data' => array('serviceId' => $rateData['serviceId']),
        'cost'      => $cost
    );

    $this->add_rate($rate);
}

}



    public function add_rate_cost($cost, $package)
    {
        $rate = apply_filters('tipax_add_rate', [
            'id'    => $this->get_rate_id(),
            'label' => $this->title,
            'cost'  => $cost,
        ], $package, $this);

        $this->add_rate($rate);
    }

    public function get_terms_option($term_id)
    {
        $ancestors = get_ancestors($term_id, 'state_city');

        if (empty($ancestors)) {
            return [];
        }

        array_unshift($ancestors, $term_id);

        $options = array_map([$this, 'get_term_option'], $ancestors);

        return $options;
    }

    public function checkApiKey($refreshToken = false)
    {
        if ($refreshToken == true) {
            $data = [
                'username' => isset($this->settings["username"]) ? $this->settings["username"] : null,
                'password' => isset($this->settings["password"]) ? $this->settings["password"] : null,
                'apiKey' => isset($this->settings["apiKey"]) ? $this->settings["apiKey"] : null,
            ];
            $response = $this->request->apiRequest('Account/token', $data, "POST");

            if ($response) {
                $accessToken = $response->accessToken;
                $this->accessToken = $accessToken;
                update_option('woocommerce_tipax_shipping_method_accesstoken', $accessToken, '', 'yes');
            }
        }

        if ((isset($this->settings['apiKey'])) || (isset($this->settings['username'])) || (isset($this->settings['password']))) {

            if ($this->settings['apiKey'] || $this->settings['username']  || $this->settings['password']) {

                $data = [
                    'username' => isset($this->settings["username"]) ? $this->settings["username"] : null,
                    'password' => isset($this->settings["password"]) ? $this->settings["password"] : null,
                    'apiKey' => isset($this->settings["apiKey"]) ? $this->settings["apiKey"] : null,
                ];


                if (isset($this->accessToken) || !$this->accessToken) {
                    if ($this->accessToken == null) {

                        $response = $this->request->apiRequest('Account/token', $data, "POST");
                        if (!isset($response->errors)) {
                            $accessToken = isset($response->accessToken) ? $response->accessToken : null;

                            if ($accessToken != null) {
                                $this->accessToken = $accessToken;
                                if (get_option('woocommerce_tipax_shipping_method_accesstoken') !== false) {
                                    update_option('woocommerce_tipax_shipping_method_accesstoken', $accessToken, '', 'yes');
                                } else {
                                    add_option('woocommerce_tipax_shipping_method_accesstoken', $accessToken, '', 'yes');
                                }
                            }
                        } else {
                            $message = new Message();
                            return $message->default($response->message);
                        }
                    }
                }
            }
        }
    }

    /**
     * Processes and saves global shipping method options in the admin area.
     *
     * This method is usually attached to woocommerce_update_options_x hooks.
     *
     * @since 2.6.0
     * @return bool was anything saved?
     */
    public function process_admin_options()
    {
        //when edit main tipax settings, address and accesstoken will be deleted
        if ($this->instance_id == 0 && $this->id == "tipax_shipping_method") {
            delete_option("woocommerce_tipax_shipping_method_accesstoken");
            tipax_log("process_admin_options: woocommerce_tipax_shipping_method_accesstoken Deleted");
            delete_transient('tipax_addresses');
            tipax_log("process_admin_options: tipax_addresses transient Deleted");
        }

        if (! $this->instance_id) {
            return parent::process_admin_options();
        }

        // Check we are processing the correct form for this instance.
        if (! isset($_REQUEST['instance_id']) || absint($_REQUEST['instance_id']) !== $this->instance_id) { // WPCS: input var ok, CSRF ok.
            return false;
        }

        $this->init_instance_settings();


        $post_data = $this->get_post_data();
        if (isset($post_data["woocommerce_tipax_shipping_method_address_method"]) && $post_data["woocommerce_tipax_shipping_method_address_method"] == 2) {
            if ($post_data['woocommerce_tipax_shipping_method_postal_code'] != $this->get_option('postal_code')) {
                $lat = explode(',', $post_data['woocommerce_tipax_shipping_method_lat_lon'])[0];
                $lon = explode(',', $post_data['woocommerce_tipax_shipping_method_lat_lon'])[1];
                $data = [
                    'title' => $post_data['woocommerce_tipax_shipping_method_title'],
                    'cityId' => $post_data['woocommerce_tipax_shipping_method_city'],
                    'mainStreet1' => $post_data['woocommerce_tipax_shipping_method_address'],
                    'latitude' => $lat,
                    'longitude' => $lon,
                    'postalCode' => $post_data['woocommerce_tipax_shipping_method_postal_code'],
                    'unit' => $post_data['woocommerce_tipax_shipping_method_unit'],
                    'floor' => $post_data['woocommerce_tipax_shipping_method_floor'],
                    'no' => $post_data['woocommerce_tipax_shipping_method_plaque'],
                    'isDefault' => true
                ];
                if ($this->accessToken) {
                    $response = $this->request->apiRequest('Addresses', $data, "POST", $this->accessToken);
                    if (get_option('tipax_originid') !== false) {
                        update_option('tipax_originid', $response->addressesId, '', 'yes');
                    } else {
                        add_option('tipax_originid', $response->addressesId, '', 'yes');
                    }
                    if (get_option('tipax_originparentid') !== false) {
                        update_option('tipax_originparentid', $response->id, '', 'yes');
                    } else {
                        add_option('tipax_originparentid', $response->id, '', 'yes');
                    }
                }
            }
        }

        foreach ($this->get_instance_form_fields() as $key => $field) {
            if ('title' !== $this->get_field_type($field)) {
                try {
                    $this->instance_settings[$key] = $this->get_field_value($key, $field, $post_data);
                } catch (Exception $e) {
                    $this->add_error($e->getMessage());
                }
            }
        }
        return update_option($this->get_instance_option_key(), apply_filters('woocommerce_shipping_' . $this->id . '_instance_settings_values', $this->instance_settings, $this), 'yes');
    }

    public function loadAddresses()
    {
        $controlCall = 0;
        if (get_transient('addresses_loaded')) {
            $controlCall = 1;
        }
        if (!$controlCall && isset($_GET['page']) && $_GET['page'] === 'wc-settings' && isset($_GET['tab']) && $_GET['tab'] === 'shipping') {
            set_transient('addresses_loaded', true, 10);
            $tipax_addresses = get_transient('tipax_addresses');
            $data = [];

            if ($this->accessToken) {
                $response = $this->request->apiRequest('Addresses', $data, "GET", $this->accessToken);
            }
            $allAddresses = [];
            if (isset($response[0])) {
                foreach ($response as $address) {
                    if ($address->isDefault) {
                        $tipax_addresses = $address;
                    }
                    $allAddresses[$address->id]['originId'] = $address->addressesId;
                    $allAddresses[$address->id]['fullAddress'] = $address;
                }
            }

            set_transient('tipax_addresses', $tipax_addresses, 12 * HOUR_IN_SECONDS);



            $addresses_objects = $tipax_addresses;
            $dataAddr = [];
            if (get_option('tipax_shipping_address') !== false) {
                $originIdToCheck = get_option('tipax_shipping_address');
            } else {
                $originIdToCheck = $tipax_addresses->addressesId;
            }
            if (count($allAddresses) > 0) {
                $exists = false;

                // Loop through each object in the array
                foreach ($allAddresses as $object) {
                    // Check if the 'originId' matches the one we're looking for
                    if ($object['originId'] == $originIdToCheck) {
                        $exists = true;
                        break; // Break the loop once the ID is found
                    }
                }
                if (!$exists) {
                    $originAddressId = $allAddresses[$addresses_objects->id]['originId'];
                    if (get_option('tipax_shipping_address') !== false) {
                        update_option('tipax_shipping_address', $originAddressId, '', 'yes');
                    } else {
                        add_option('tipax_shipping_address', $originAddressId, '', 'yes');
                    }
                    delete_transient('tipax_addresses');
                    set_transient('tipax_addresses', $allAddresses[$addresses_objects->id]['fullAddress'], 12 * HOUR_IN_SECONDS);
                }
            }
        }
        $tipax_addresses = get_transient('tipax_addresses');

        //after one day just update the addresses
        //if there is no cache
        //if is not array it means UnauthorizedException happened
        if (false === $tipax_addresses || !isset($tipax_addresses->address)) {
            $data = [];
            if ($this->accessToken) {
                $response = $this->request->apiRequest('Addresses', $data, "GET", $this->accessToken);
            }

            if (isset($response[0])) {
                foreach ($response as $address) {
                    if ($address->isDefault) {
                        $tipax_addresses = $address;
                    }
                }
            }

            set_transient('tipax_addresses', $tipax_addresses, 12 * HOUR_IN_SECONDS);
        }
        return $tipax_addresses;
    }
    public function loadCities()
    {
        $tipax_cities = get_transient('tipax_cities');
        if (false === $tipax_cities ||  count($tipax_cities) == 0) {
            $tipax_cities = [];
            $responses = [];
            $data = [];
            if ($this->accessToken) {
                $responses = $this->request->apiRequest('Cities/plusstate', $data, "GET", $this->accessToken);
            }
            if (isset($responses->errors)) {
                return false;
            }
            if (isset($responses) && is_array($responses)) {
                foreach ($responses as $response) {
                    $tipax_cities[$response->id] = $response->title . ' - ' . $response->stateTitle;
                }
            }
            set_transient('tipax_cities', $tipax_cities, 12 * HOUR_IN_SECONDS);
        }
        return $tipax_cities;
    }

    public function loadCustomerInfo()
    {
        $data = [];
        if ($this->accessToken) {
            $response = $this->request->apiRequest('Customers', $data, "GET", $this->accessToken);
        }
        return $response;
    }
}
